// poly-mapPV.cxx
/*-------------------------------------------------------------------------
  MapMaker.cxx

  Written by Per Liedman, started February 2000.
  Based on a perl-script written by Alexei Novikov (anovikov@heron.itep.ru)

  Copyright (C) 2000 Per Liedman, liedman@home.se

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
---------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <iostream>
#include <math.h>
#include <zlib.h>
#ifndef _MSC_VER
#  include <unistd.h>
#endif
#include <sys/stat.h>
#include <plib/ul.h>

//#include "MapMaker.hxx"
#include "poly-map.hxx"
#ifdef POLY_VIEW
// ===========================================================
#include "lib_sprtf.hxx"
#include "poly-utils.hxx"
#include "poly-load.hxx"    // for VERB9
#include <sstream>
using std::ostringstream;
#undef printf
#define  printf sprtf

#ifdef NDEBUG
#define sprtf_pm
#define sprtf_mappv
#else
#define sprtf_pm sprtf
#define sprtf_mappv sprtf
#endif

// ===========================================================
#else // !POLY_VIEW
// ===========================================================
using std::cout;
// ===========================================================
#endif // POLY_VIEW y/n

/*#include <simgear/magvar/magvar.hxx>*/
static int show_dbg_set_color = 0;  // VERB9: output color change

#ifdef POLY_VIEW
// ==================================================================
/* Just for POLY-VIEW
   The MAIN thing is this is NOT drawing anything, but just accumuating
   the 'paint' points/objects ready for a later PAINT!

   SO there is NOT 'conversion' using scale, size, zoom at this
   stage. All that will be done at the actual PAINT time.

   This is intended to be run on a separate THREAD, and ONLY
   when that thread is complete, will the actual PAINT take
   place.

   In WIN32 the points/object are accumulated in WSG84 lat,lon degrees
   ------------------------------------------------------------------ */
static int in_all_one_color = 1; // can not yet SPLIT this triangle

// to copy from a DOUBLE to a FLOAT
#define sgCopyVec3d(a, b) { \
   sgVec3 _v; _v[0] = b[0]; _v[1] = b[1]; _v[2] = b[2];\
   sgCopyVec3(a,_v); }

void sgSubVec3d ( Point3D & dst, const Point3D src1, const Point3D src2 )
{
  dst [ 0 ] = src1 [ 0 ] - src2 [ 0 ] ;
  dst [ 1 ] = src1 [ 1 ] - src2 [ 1 ] ;
  dst [ 2 ] = src1 [ 2 ] - src2 [ 2 ] ;
}

void sgAddVec3d ( Point3D & dst, const Point3D src )
{
  dst [ 0 ] += src [ 0 ] ;
  dst [ 1 ] += src [ 1 ] ;
  dst [ 2 ] += src [ 2 ] ;
}

void MapMaker::pv_create_sub_point(float *topVert, float *bottomVert, 
				float *topNorm, float *bottomNorm, 
				int dest, double elevation,
				sgVec3 *ts, sgVec3 *nrms, sgVec3 *ps)
{
    sgVec3 newPoint, newNorm;
    double scaling = (elevation - bottomVert[2]) / (topVert[2] - bottomVert[2]);
    assert((scaling <= 1.0) && (0.0 <= scaling));

    sgSubVec3(newPoint, topVert, bottomVert);
    sgScaleVec3(newPoint, scaling);
    sgAddVec3(newPoint, bottomVert);

    sgSubVec3(newNorm, topNorm, bottomNorm);
    sgScaleVec3(newNorm, scaling);
    sgAddVec3(newNorm, bottomNorm);

    sgCopyVec3(ts[dest], newPoint);
    sgCopyVec3(nrms[dest], newNorm);

    //sgSetVec3(ps[dest],
    //   newPoint[0],
    //   newPoint[1],
    //   newPoint[2]);
    sgSetVec3(ps[dest],
       newPoint[0],
       newPoint[1],
       ((topVert[2] + bottomVert[2]) / 2.0));
}



void MapMaker::pv_split_elevation_tri(int vert0, int vert1, int vert2,
                                     int norm0, int norm1, int norm2,
                                     vector<float*> &v, vector<float*> &n,
                                     vector<Point3D> &vp, vector<Point3D> &np,
                                     int col)
{
   bool smooth = (features & DO_SMOOTH_COLOR) ? true : false;
   sgVec3 t[3], nrm[3];
   sgVec2 p[3];
   int index[3];

   // copy the verts
   sgCopyVec3(t[0], v[vert0]);
   sgCopyVec3(t[1], v[vert1]);
   sgCopyVec3(t[2], v[vert2]);
   // copy the norms
   sgCopyVec3(nrm[0], n[norm0]);
   sgCopyVec3(nrm[1], n[norm1]);
   sgCopyVec3(nrm[2], n[norm2]);

   index[0] = elev2index(t[0][2]);
   index[1] = elev2index(t[1][2]);
   index[2] = elev2index(t[2][2]);

  // Triangle lies within one elevation level.  Draw it in one
  // colour.
  if ((index[0] == index[1]) && (index[1] == index[2])) {
     sprtf("ERROR: pv_split_elevation_tri called with EQUAL INDEXES!\n");
     return;
  }

  // Triangle spans more than one level.  Drats.  Do a quick sort on
  // the vertices, so that vert0 points to the top vertex, vert1,
  // the middle, and vert2 the bottom.
  if (index[0] < index[1]) {
     swap(vert0, vert1);
     swap(norm0, norm1);
     swap(index[0], index[1]);
  }
  if (index[0] < index[2]) {
     swap(vert0, vert2);
     swap(norm0, norm2);
     swap(index[0], index[2]);
  }

  if (index[1] < index[2]) {
     swap(vert1, vert2);
     swap(norm1, norm2);
     swap(index[1], index[2]);
  }

  // Now begin slicing the lines leading away from vert0 to vert1
  // and vert2.  Slicing a triangle create new triangles, new
  // quadrilaterals, and even new pentagons.  Because the triangle
  // lies in a plane (by definition), we are assured that the new
  // figures are also planar.
  //
  // After each bit is sliced off the top, it is drawn and then
  // discarded.  The process is then repeated on the remaining
  // figure, until there's nothing left.
  //
  // This can be illustrated with the power of ASCII graphics.  If
  // we have to make two cuts of the triangle ABC, we'll create 4
  // new points, D, E, F, and G.  This creates 3 figures: ADE,
  // EDBFG, and GFC.
  //
  //        A	       A	  	 
  //       /|	      /|	  	 
  //      / |	     / |	  	 
  //  -->/  |	    D--E     D--E      D--E
  //    /   |	   /   |    /   |     /   |
  //   B    |	  B    |   B    |    B    |
  //    \   |	   \   |    \   |     \   |
  //     \  |	    \  |     \  |      \  |
  //   -->\ |	  -->\ |   -->\ |       F-G       F-G
  //       \|	      \|       \|        \|        \|
  //        C	       C        C         C         C

  int k, vertices;
  sgVec3 ts[5], nrms[5];
  sgVec2 ps[5];
  sgVec3 ps3[5];

  // Slicing creates new vertices and normals, so we need to keep
  // track of actual points, not just their indices as before.  The
  // array 'ts' keeps the vertices of the current figure, 'nrms' the
  // norms, and 'ps' the scaled points.  At most we can generate a
  // pentagon, so each array has 5 points.  The current number of
  // points is given by 'vertices'.
  //
  // In the example above, the arrays will contain data for ADE,
  // then DEBFG, and finally GFC.  Note that points are given in a
  // counter-clockwise direction (as illustrated here).  This means
  // that that when a bottom line (eg, DE in ADE) becomes a top line
  // (ED in EDBFG), we reverse its order.

  sgCopyVec3(ts[0], v[vert0]);   // copy 3 xyz
  sgCopyVec3(nrms[0], n[norm0]);
  sgSetVec2(ps[0], 
     ts[0][0],
     ts[0][1]);

  // set 0
  sgSetVec3(ps3[0], // keep 3 xyz, for conversion
	      ts[0][0], 
	      ts[0][1],
         ts[0][2]);

  vertices = 1;
  for (k = index[0]; k > index[1]; k--) {
     // Make a cut and draw the resulting figure.
     double elevation = elev_height[k - 1]; // This is where we cut.
     // Cut along the short line (vert0 to vert1), and put the
     // resulting vertex, normal, and point into ts, nrms, and ps.
     create_sub_point(v[vert0], v[vert1], n[norm0], n[norm1], 
        vertices, elevation, ts, nrms, ps);  // set 1
     pv_create_sub_point(v[vert0], v[vert1], n[norm0], n[norm1], 
        vertices++, elevation, ts, nrms, ps3);  // set 1
     // Ditto for the long line (vert0 to vert2).
     create_sub_point(v[vert0], v[vert2], n[norm0], n[norm2], 
        vertices, elevation, ts, nrms, ps);  // set 2
     pv_create_sub_point(v[vert0], v[vert2], n[norm0], n[norm2], 
        vertices++, elevation, ts, nrms, ps3);  // set 2

     // Now draw the resulting figure.
     pv_draw_elevation_slice(vertices, smooth, k, ts, nrms, ps3);

     // We're ready to move down and make the next slice.  The two
     // points we just created will now be the top of the next
     // slice.  We need to reverse the order of the points.
     sgCopyVec3(ts[0], ts[vertices - 1]);
     sgCopyVec3(nrms[0], nrms[vertices - 1]);
	  sgCopyVec2(ps[0], ps[vertices - 1]);

	  sgCopyVec3(ps3[0], ps[vertices - 1]);

     sgCopyVec3(ts[1], ts[vertices - 2]);
     sgCopyVec3(nrms[1], nrms[vertices - 2]);
     sgCopyVec2(ps[1], ps[vertices - 2]);

	  sgCopyVec3(ps3[1], ps[vertices - 2]);

     vertices = 2;
  }

  // Add the middle vertex.
  sgCopyVec3(ts[vertices], v[vert1]);
  sgCopyVec3(nrms[vertices], n[norm1]);
  sgSetVec2(ps[vertices],
     ts[vertices][0],
     ts[vertices][1]);

  sgSetVec3(ps3[vertices],
     ts[vertices][0],
     ts[vertices][1],
     ts[vertices][2] );

  vertices++;
  assert(vertices <= 5);

  for (; k > index[2]; k--) {
     // Make a cut and draw the resulting figure.
     double elevation = elev_height[k - 1]; // This is where we cut.

     // Get the point along the short line.
     create_sub_point(v[vert1], v[vert2], n[norm1], n[norm2], 
        vertices, elevation, ts, nrms, ps);
     pv_create_sub_point(v[vert1], v[vert2], n[norm1], n[norm2], 
        vertices++, elevation, ts, nrms, ps3);

     // Get the point along the long line.
     create_sub_point(v[vert0], v[vert2], n[norm0], n[norm2], 
        vertices, elevation, ts, nrms, ps);
     pv_create_sub_point(v[vert0], v[vert2], n[norm0], n[norm2], 
        vertices++, elevation, ts, nrms, ps3);

     pv_draw_elevation_slice(vertices, smooth, k, ts, nrms, ps3);

     // The bottom will be the next top.
     sgCopyVec3(ts[0], ts[vertices - 1]);
     sgCopyVec3(nrms[0], nrms[vertices - 1]);
     sgCopyVec2(ps[0], ps[vertices - 1]);
     
     sgCopyVec3(ps3[0], ps3[vertices - 1]);

     sgCopyVec3(ts[1], ts[vertices - 2]);
     sgCopyVec3(nrms[1], nrms[vertices - 2]);
     sgCopyVec2(ps[1], ps[vertices - 2]);
	
     sgCopyVec3(ps3[1], ps3[vertices - 2]);

     vertices = 2;
  }

  // Add the final vertex and draw the last figure.
  sgCopyVec3(ts[vertices], v[vert2]);
  sgCopyVec3(nrms[vertices], n[norm2]);
  sgSetVec2(ps[vertices], 
     ts[vertices][0], 
     ts[vertices][1]);

  sgSetVec3(ps3[vertices], 
     ts[vertices][0],
     ts[vertices][1],
     ts[vertices][2]);

  vertices++;
  assert(vertices <= 5);

  pv_draw_elevation_slice(vertices, smooth, k, ts, nrms, ps3);

}

void MapMaker::pv_draw_elevation_slice(int vertices, bool smooth, int k,
				    sgVec3 *ts, sgVec3 *nrms, sgVec3 *ps3)
{
   sgVec4 color[5];
   sgVec2 ps[5];
   int i;
   if (smooth) {
      for (int i = 0; i < vertices; i++) {
         elev2colour_smooth((int)ts[i][2], color[i]);
      }
   }

   sprtf("WARNING: Converting to WGS84 not yet CHECKED!\n");
   //return;
   // hmmm, sgVec2 *ps points to 
   // a pair of triangles,
   // or a quad plus a triangle,
   // or a pentagon, but still draw a quad plus a triangle
   // so we have ps3[0], ps3[1], ps3[2], ps3[3] and ps3[4] to convert
   // ==========================================================
   Point3D pt3d[5];
   SGVec3d vert; // = ps3[i];
   for( i = 0; i < 5; i++ ) {
      vert[0] = ps3[i][0];
      vert[1] = ps3[i][1];
      vert[2] = ps3[i][2];
      pt3d[i] = cartv3d_2_p3dllh( vert, gbs_cent );
      ps[i][0] = pt3d[i].lon();
      ps[i][1] = pt3d[i].lat();
      // NO NO sgVec2 ps[i][2] = pt3d[i].elev();
   }
   // Draw the figure.
   if (vertices == 3) {
      // Triangle
      sprtf_mappv("mappv:Tri: %s(%s), %s(%s), %s(%s)\n",
         sgv22stg(ps[0]),
         (sgv2_out_of_bucket_range(ps[0], tile_bucket) ? "OOR" : "ok"),
         sgv22stg(ps[1]),
         (sgv2_out_of_bucket_range(ps[1], tile_bucket) ? "OOR" : "ok"),
         sgv22stg(ps[2]),
         (sgv2_out_of_bucket_range(ps[2], tile_bucket) ? "OOR" : "ok"));
      if (smooth) {
         output->drawTriangle(ps, nrms, color);
      } else {
         output->setColor(mm_palette[elev_colindex[k]]);
         output->drawTriangle(ps, nrms);
      }
   } else if (vertices == 4) {
      // Quadrilateral.
      sprtf_mappv("mappv:Quad: %s(%s), %s(%s), %s(%s), %s(%s)\n",
         sgv22stg(ps[0]),  // pt3d2stg(pt3d[0]),
         (sgv2_out_of_bucket_range(ps[0], tile_bucket) ? "OOR" : "ok"),
         sgv22stg(ps[1]),  // pt3d2stg(pt3d[1]),
         (sgv2_out_of_bucket_range(ps[1], tile_bucket) ? "OOR" : "ok"),
         sgv22stg(ps[2]),  // pt3d2stg(pt3d[2]),
         (sgv2_out_of_bucket_range(ps[2], tile_bucket) ? "OOR" : "ok"),
         sgv22stg(ps[3]), // pt3d2stg(pt3d[3]));
         (sgv2_out_of_bucket_range(ps[3], tile_bucket) ? "OOR" : "ok"));
      if (smooth) {
         output->drawQuad(ps, nrms, color);
      } else {
         output->setColor(mm_palette[elev_colindex[k]]);
         output->drawQuad(ps, nrms);
      }
   } else {
      // Pentagon.  Draw it as a quadrilateral and a triangle.  The
      // quadralateral consists of the first 4 points, and the
      // triangle consists of the remaining two *and* the first one.
      // To draw the triangle, then, we copy the last two points
      // over the second and third points (this is okay, because
      // they won't be used again), and draw a triangle consisting
      // of the first 3 points.
      sprtf_mappv("mappv:PQuad: %s(%s), %s(%s), %s(%s), %s(%s)\n",
         sgv22stg(ps[0]),  // pt3d2stg(pt3d[0]),
         (sgv2_out_of_bucket_range(ps[0], tile_bucket) ? "OOR" : "ok"),
         sgv22stg(ps[1]),  // pt3d2stg(pt3d[1]),
         (sgv2_out_of_bucket_range(ps[1], tile_bucket) ? "OOR" : "ok"),
         sgv22stg(ps[2]),  // pt3d2stg(pt3d[2]),
         (sgv2_out_of_bucket_range(ps[2], tile_bucket) ? "OOR" : "ok"),
         sgv22stg(ps[3]), // pt3d2stg(pt3d[3]));
         (sgv2_out_of_bucket_range(ps[3], tile_bucket) ? "OOR" : "ok"));
      if (smooth) {
         output->drawQuad(ps, nrms, color);
      } else {
         output->setColor(mm_palette[elev_colindex[k]]);
         output->drawQuad(ps, nrms);
      }
   }

   sgCopyVec3(ts[1], ts[3]);
   sgCopyVec3(nrms[1], nrms[3]);
   sgCopyVec2(ps[1], ps[3]);
   
	sgCopyVec3(ts[2], ts[4]);
	sgCopyVec3(nrms[2], nrms[4]);
	sgCopyVec2(ps[2], ps[4]);

   sprtf_mappv("mappv:LTri: %s(%s), %s(%s), %s(%s)\n",
      sgv22stg(ps[0]),
      (sgv2_out_of_bucket_range(ps[0], tile_bucket) ? "OOR" : "ok"),
      sgv22stg(ps[1]),
      (sgv2_out_of_bucket_range(ps[1], tile_bucket) ? "OOR" : "ok"),
      sgv22stg(ps[2]),
      (sgv2_out_of_bucket_range(ps[2], tile_bucket) ? "OOR" : "ok"));
	if (smooth) {
      sgCopyVec4(color[1], color[3]);
      sgCopyVec4(color[2], color[4]);
      output->drawTriangle(ps, nrms, color);
   } else {
      output->drawTriangle(ps, nrms);
   }
}

void MapMaker::pv_draw_elevation_tri(int vert0, int vert1, int vert2,
                                     int norm0, int norm1, int norm2,
                                     vector<float*> &v, vector<float*> &n,
                                     vector<Point3D> &vp, vector<Point3D> &np,
                                     int col)
{
   bool smooth = (features & DO_SMOOTH_COLOR) ? true : false;
   sgVec3 t[3], nrm[3];
   sgVec2 p[3];
   sgVec3 t3[3];
   int index[3];

   // copy verts
   sgCopyVec3(t[0], v[vert0]);
   sgCopyVec3(t[1], v[vert1]);
   sgCopyVec3(t[2], v[vert2]);
   sgCopyVec3(nrm[0], n[norm0]);
   sgCopyVec3(nrm[1], n[norm1]);
   sgCopyVec3(nrm[2], n[norm2]);
   index[0] = elev2index(t[0][2]);
   index[1] = elev2index(t[1][2]);
   index[2] = elev2index(t[2][2]);
   if( !in_all_one_color &&
      ( (index[0] != index[1]) || (index[1] != index[2]) ) ) {
      int  save_size = size;
      float save_zoom = zoom; // size / scale
      size = 0;
      zoom = 1.0;
      sprtf("WARNING: SPLIT NOT YET CHECKED! index 1=%d, 2=%d, 3=%d\n",
       index[0], index[1], index[2] );

      pv_split_elevation_tri(vert0, vert1, vert2,
       norm0, norm1, norm2,
       v, n,
       vp, np, col);

      size = save_size;
      zoom = save_zoom;
      return;
   }

   if ( (index[0] != index[1]) || (index[1] != index[2]) ) {
       static int done_warning = 0;
       if ( !done_warning )
           sprtf("\nWARNING: Split of this/these trianges NOT CODED!\n\n" );
       done_warning++;
   }

   // Triangle lies within one elevation level.  Draw it in one
   // colour.
   // copy wgs84 coordinates
   sgCopyVec3d(t3[0], vp[vert0]);
   sgCopyVec3d(t3[1], vp[vert1]);
   sgCopyVec3d(t3[2], vp[vert2]);

       // copy actual wgs84 coordinates
	sgSetVec2(p[0], t3[0][0], t3[0][1]);
	sgSetVec2(p[1], t3[1][0], t3[1][1]);
	sgSetVec2(p[2], t3[2][0], t3[2][1]);
	if (smooth) {
	    sgVec4 color[3];
	    elev2colour_smooth((int)t[0][2], color[0]);
	    elev2colour_smooth((int)t[1][2], color[1]);
	    elev2colour_smooth((int)t[2][2], color[2]);
	    output->drawTriangle( p, nrm, color );
	} else {
	    output->setColor(mm_palette[elev2colour(v[vert0][2])]);
	    output->drawTriangle( p, nrm );
	}
}

void MapMaker::pv_draw_a_tri(int vert0, int vert1, int vert2,
			  int norm0, int norm1, int norm2,
           vector<float*> &v, vector<float*> &n, 
			  vector<Point3D> &vp, vector<Point3D> &np, int col)
{
   bool save_shade = output->getShade();
    
   // Elevation triangles get special treatment.
   if (col == -1) {
      pv_draw_elevation_tri(vert0, vert1, vert2, norm0, norm1, norm2, v, n, vp, np, col);
      return;
   }

   // Non-elevation triangles are coloured according to col.  They
   // are shaded (usually), but not smoothed.
   output->setColor(mm_palette[col]);
   if (show_dbg_set_color && VERB9)
       sprtf("[V9]:pv_draw_a_tri: setColor %d %s\n", col, float3p2rgbstg(mm_palette[col],0));
   if (col == 12 || col == 13) {
      // do not shade ocean/lake/etc.
      // DCL - and switch lighting off for water for now (see the Jan 2005 mailing list archives)
      output->setShade(false);
   }

   sgVec3 nrm[3];
   sgVec2 p[3];
    
   sgCopyVec3(nrm[0], n[norm0]);
   sgCopyVec3(nrm[1], n[norm1]);
   sgCopyVec3(nrm[2], n[norm2]);

   sgSetVec2(p[0], 
	      vp[vert0][0], 
	      vp[vert0][1]);
   sgSetVec2(p[1], 
	      vp[vert1][0], 
	      vp[vert1][1]);
    sgSetVec2(p[2], 
	      vp[vert2][0], 
	      vp[vert2][1]);

    output->drawTriangle(p, nrm);
    
    // DCL - restore the original lighting in case we turned it off for water
    output->setShade(save_shade);
}

void MapMaker::pv_draw_pts(const int_list &vertex_indices, 
			const int_list &normal_indices, 
         vector<float*> &v, vector<float*> &n, 
			vector<Point3D> &vp, vector<Point3D> &np, int col) 
{
   int i;
   int vert0, norm0;
   sgVec3 nrm[3];
   sgVec2 p[3];

   for (i = 0; i < vertex_indices.size(); i++) {
      vert0 = vertex_indices[i];
      norm0 = normal_indices[i];
      sgSetVec2(p[0], 
	      vp[vert0][0], 
	      vp[vert0][1]);
      sgSetVec2(p[1], 
	      vp[vert0][0], 
	      vp[vert0][1]);
      sgSetVec2(p[2], 
	      vp[vert0][0], 
	      vp[vert0][1]);
      sgCopyVec3(nrm[0], n[norm0]);
      sgCopyVec3(nrm[1], n[norm0]);
      sgCopyVec3(nrm[2], n[norm0]);
      output->drawPoint(p, nrm);
   }
}

// ============================================
// TRIANGLES - 3 vertices, in one plane
// v0   v1
// ------
// |    /
// |   /
// |  /
// | /
// |/
// v3
// ============================================
void MapMaker::pv_draw_tris(const int_list &vertex_indices, 
			const int_list &normal_indices, 
         vector<float*> &v, vector<float*> &n, 
			vector<Point3D> &vp, vector<Point3D> &np, int col) 
{
   int i;
   int vert0, vert1, vert2;
   int norm0, norm1, norm2;

   // EYE - can we assume indices.size() is divisible by 3?
   assert((vertex_indices.size() % 3) == 0);
   for (i = 0; i < vertex_indices.size(); i += 3) {
      vert0 = vertex_indices[i];
      norm0 = normal_indices[i];
      vert1 = vertex_indices[i + 1];
      norm1 = normal_indices[i + 1];
      vert2 = vertex_indices[i + 2];
      norm2 = normal_indices[i + 2];
      pv_draw_a_tri(vert0, vert1, vert2, norm0, norm1, norm2, v, n, vp, np, col);
   }
}

// ======================================================
// TRI_FAN - TRIANGLES, FANNING AROUND A SINGLE VERTEX
// v1
// ------ v2
// |   /|
// |  / | 
// | /  |
// |/   |
// v0---| v3
//   \  |
//    \ |
//     \|
//     v4
// =====================================================

void MapMaker::pv_draw_trifan(const int_list &vertex_indices, 
			   const int_list &normal_indices, 
			   vector<float*> &v, vector<float*> &n,
			vector<Point3D> &vp, vector<Point3D> &np, int col) 
{
    int i;
    int cvert, vert1, vert2;
    int cnorm, norm1, norm2;

    cvert = vertex_indices[0];
    cnorm = normal_indices[0];
    vert1 = vertex_indices[1];
    norm1 = normal_indices[1];
    for (i = 2; i < vertex_indices.size(); i++) {
	vert2 = vertex_indices[i];
	norm2 = normal_indices[i];

	pv_draw_a_tri(cvert, vert1, vert2, cnorm, norm1, norm2, v, n, vp, np, col);

	vert1 = vert2;
	norm1 = norm2;
    }
}

// ==========================================================
// TRI_STRIP - AJOINING TRIANGES SHARING AN EDGE
// v0   v1
// ------
// |   /|\
// |  / | \ v4
// | /  | /\
// |/   |/  \
// ---------- v5
// v2   v3
// ==========================================================
void MapMaker::pv_draw_tristrip(const int_list &vertex_indices, 
			     const int_list &normal_indices, 
			     vector<float*> &v, vector<float*> &n,
              vector<Point3D> &vp, vector<Point3D> &np,
			     int col) 
{
    int i;
    int vert0, vert1, vert2;
    int norm0, norm1, norm2;
    vert0 = vertex_indices[0];
    norm0 = normal_indices[0];
    vert1 = vertex_indices[1];
    norm1 = normal_indices[1];
    for (i = 2; i < vertex_indices.size(); i++) {
       vert2 = vertex_indices[i];
       norm2 = normal_indices[i];
       pv_draw_a_tri(vert0, vert1, vert2, norm0, norm1, norm2, v, n, vp, np, col);
       vert1 = vert0;
       norm1 = norm0;
       vert0 = vert2;
       norm0 = norm2;
    }
}

Point3D MapMaker::cartv3d_2_p3dllh( SGVec3d cart, const SGVec3d &c )
{
   cart += c;
   SGGeod geod;
   SGGeodesy::SGCartToGeod( cart, geod );
   Point3D pt( geod.getLongitudeDeg(), geod.getLatitudeDeg(), geod.getElevationM() );
   return pt;
}

// ==================================================================
#endif // POLY-VIEW

// eof - poly-mapPV.cxx
