// poly-load.cxx
// load-poly.cpp -- deal with loaded set of poly files
//
// Written by Geoff McLane, started May, 2009.
//
// Copyright (C) 2009  Geoff R. McLane  - http://geoffair.net/fg
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// $Id: main.cxx,v 1.58 2005-09-28 16:40:32 curt Exp $


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <simgear/compiler.h>

#include <iostream>
#include <string>
#include <vector>

#include <plib/sg.h>

#include <simgear/constants.h>

#include "load-poly.hxx"
#include "poly-load.hxx"
#include "poly-utils.hxx"
#include "lib_sprtf.hxx"
#include "poly-ini.hxx"

using std::cout;
using std::cerr;
using std::endl;
using std::string;
using std::vector;

extern void set_mapobj_mapsize( int size );  // set the mapobj size

// ==========================================================
TGPolyLoad * active_polyload = NULL;
vector<TGPolyLoad *> loaded_polys;
vector<int> enabled_polys;
static PolyGlobs polyglobs;   // global parameters

static int next_loaded_poly = 0;
static int current_loaded_poly = 0;
int get_loaded_poly_count( void ) { return loaded_polys.size(); }

inline PPolyGlobs get_poly_globs(void) { return &polyglobs; }

void init_poly_load(void)
{
   PPolyGlobs ppg = get_poly_globs();
   ppg->g_verbosity = 0;            // --verbosity=<num>
   ppg->g_map_size = 600.0;
   ppg->g_map_zoom = DEFAULT_ZOOM;
   ppg->g_pixel_move_x = 5;        // default right/left
   ppg->g_pixel_move_y = 5;        // default up/down
   ppg->g_offset_change_x = 0.1;   // this should vary per zoom level
   ppg->g_offset_change_y = 0.1;   // and per the above pixel move item
   ppg->g_offset_lat = 0.0;
   ppg->g_offset_lon = 0.0;
   ppg->g_set_c_lat = 0.0;
   ppg->g_set_c_lon = 0.0;
   ppg->g_zoom_change = DEFAULT_ZOOM_CHANGE;
   //ppg->g_c_lat = ppg->g_offset_lat + ppg->g_set_c_lat;
   //ppg->g_c_lon = ppg->g_offset_lon + ppg->g_set_c_lon;
   ppg->g_auto_zoom = 1; // adjust zoom value so range will fill screen
   ppg->g_join_poly_points = 1;  // set default
   ppg->g_join_first_to_last = 1;
   ppg->g_paint_fit_points = 0; // --[enable|disable]-paint-fit-points - set default
   ppg->g_paint_fit_tris = 0;   // --[enable|diable\-paint-fit-tris - set default
   ppg->g_paint_bounding_box = 0; // paint_bounding_box( hdc, &mm ); can overwrite points!
   ppg->g_paint_airports = 1;     // only 1 or 2 selected - could load apt.dat...
   ppg->g_paint_colors = 1;       // use a set of colors
   ppg->g_min_lon = -180.0; // these are set in Out_Poly_Counts()
   ppg->g_max_lon =  180.0; // after completion of
   ppg->g_min_lat =  -90.0; // the poly loading
   ppg->g_max_lat =   90.0; // representling 1 or more polys
   ppg->g_xdist   =  180.0;
   ppg->g_ydist   =   90.0;
   ppg->g_fill_polygon = 0;   // 'f' toggle
   ppg->g_fill_scene = 1;  // 'F' toggle
   ppg->g_got_fg_scenery = 0;
   ppg->g_use_fg_scenery = 1;   // default to ON --disable-scenery --enable-scenery
   ppg->g_enable_scenery = 1;   // default to ON --disable-scenery-paint --enable-scenery-paint
   ppg->g_enable_scenery_wsg84 = 0;  // --enable-scenery-wsg84
   ppg->g_enable_scenery_normals = 0;  // --enable-scenery-normals
   ppg->g_fg_scenery = "";
   ppg->g_show_point_paint = 0;
   ppg->g_page_key_steps = giPageStep; // number of steps per PAGE UP or DOWN keys
   ppg->g_add_neighbors = 0;  // default --disable-neighbors - load nearby buckets
   ppg->g_user_center = 0;  // set if --center-lon-lat=<float,float> given
   ppg->g_paint_X = 0; // set to paint a BIG X for each POINT
   ppg->g_fixed_spread = 0;  // if paint X, and this has value, use this
   ppg->g_var_line_size = 1; // vary the line size when joining lines
}

int get_page_key_steps(void) { return get_poly_globs()->g_page_key_steps; }

void toggle_int( int * pint ) { if ( *pint ) *pint = 0; else *pint = 1; }

void toggle_fill_poly(void) { toggle_int( &get_poly_globs()->g_fill_polygon ); }
void toggle_fill_poly2(void) { toggle_int( &get_poly_globs()->g_fill_scene ); }
void toggle_paint_x(void) { toggle_int( &get_poly_globs()->g_paint_X ); }
void toggle_join_points(void) { toggle_int( &get_poly_globs()->g_join_poly_points ); }
void toggle_join_first_to_last(void) { toggle_int( &get_poly_globs()->g_join_first_to_last ); }
void toggle_paint_fit_points(void) { toggle_int( &get_poly_globs()->g_paint_fit_points ); }
void toggle_paint_fit_tris(void) { toggle_int( &get_poly_globs()->g_paint_fit_tris ); }
void toggle_scenery_paint(void) { toggle_int( &get_poly_globs()->g_enable_scenery ); }

void calculate_scroll_sizes( void )
{
   PPolyGlobs ppg = get_poly_globs();
   double mapsize, zoom, c_lat, c_lon, px, py;
   int   x, y;
   double nlat, nlon, tlat, tlon;
   // using ppg->g_pixel_move_x = 5;        // default right/left
   // and ppg->g_pixel_move_y = 5;        // default up/down
   // establish unit of lef/right
   mapsize = get_g_map_size(); // get current size
   zoom = get_g_map_zoom();    // get current zoom
   tlat = c_lat = get_g_center_lat();
   tlon = c_lon = get_g_center_lon();
   win_ll2pt( tlat, tlon, c_lat, c_lon, mapsize, zoom, &px, &py );
   x = win_int(px);
   y = win_int(py);
   // set the x change factor
   win_pt2ll( &nlat, &nlon, c_lat, c_lon, mapsize, zoom,
      (double)(x - ppg->g_pixel_move_x), (double)y );
   tlon = nlon;
   win_pt2ll( &nlat, &nlon, c_lat, c_lon, mapsize, zoom,
      (double)(x + ppg->g_pixel_move_x), (double)y );
   ppg->g_offset_change_x = abs(nlon - tlon);

   win_pt2ll( &nlat, &nlon, c_lat, c_lon, mapsize, zoom,
      (double)x, (double)(y - ppg->g_pixel_move_y) );
   tlat = nlat;
   win_pt2ll( &nlat, &nlon, c_lat, c_lon, mapsize, zoom,
      (double)x, (double)(y  + ppg->g_pixel_move_y) );
   ppg->g_offset_change_y = abs(nlat - tlat);
}

void Increment_ZOOM( void ) 
{ 
   PPolyGlobs ppg = get_poly_globs();
   ppg->g_map_zoom += ppg->g_zoom_change;
   calculate_scroll_sizes();
}
void Decement_ZOOM( void )
{
   PPolyGlobs ppg = get_poly_globs();
   double zoom = ppg->g_map_zoom;
   zoom -= ppg->g_zoom_change;
   if ( zoom <= 0.0 ) {
      zoom = ppg->g_map_zoom;
      zoom = (zoom - (zoom * 0.1));
      //if( zoom < 0.0002 )
      //    zoom = ppg->g_map_zoom; //0.00002;
   }
   ppg->g_map_zoom = zoom;

   calculate_scroll_sizes();
}

void Restore_DEFAULT_ZOOM( void )   
{
   get_poly_globs()->g_map_zoom = DEFAULT_ZOOM;
   calculate_scroll_sizes();
}
static int got_prev_zoom = 0;
void Restore_ZOOM( void )   
{
   PPolyGlobs ppg = get_poly_globs();
   if(got_prev_zoom)
      ppg->g_map_zoom = ppg->g_prev_map_zoom;
   else
      ppg->g_map_zoom = DEFAULT_ZOOM;
   calculate_scroll_sizes();
}

double get_g_map_size( void )    { return get_poly_globs()->g_map_size; }
double set_g_map_size( double size )
{
   PPolyGlobs ppg = get_poly_globs();
   double prev = ppg->g_map_size;
   ppg->g_map_size = size;
   set_mapobj_mapsize(win_int(size));
   return prev;
}
double get_g_map_zoom( void )    { return get_poly_globs()->g_map_zoom; }
double set_g_map_zoom( double zoom )
{
   PPolyGlobs ppg = get_poly_globs();
   ppg->g_prev_map_zoom = ppg->g_map_zoom;
   ppg->g_map_zoom = zoom;
   got_prev_zoom = 1;
   return ppg->g_prev_map_zoom;
}

double get_g_zoom_change( void ) { return get_poly_globs()->g_zoom_change; }
double get_g_center_lon( void )  { return get_poly_globs()->g_c_lon; }
double get_g_center_lat( void )  { return get_poly_globs()->g_c_lat; }
void get_g_center( double * plon, double * plat ) {
    *plon = get_g_center_lon();
    *plat = get_g_center_lat();
}

void set_center_lon_ppg( PPolyGlobs ppg ) {
   double lon = ppg->g_offset_lon + ppg->g_set_c_lon;
   if( lon > 180.0 )
      lon -= 360.0;
   if( lon < -180.0 )
      lon += 360.0;
   ppg->g_c_lon = lon;
}

void set_center_lat_ppg( PPolyGlobs ppg ) {
   double lat = ppg->g_offset_lat + ppg->g_set_c_lat;
   if( lat > 90.0 )
      lat = 90.0;
   if( lat < -90.0 )
      lat = -90.0;
   ppg->g_c_lat = lat;
}

void change_lon_offset( int add ) {
   PPolyGlobs ppg = get_poly_globs();
   if(add)
      ppg->g_offset_lon += ppg->g_offset_change_x;
   else
      ppg->g_offset_lon -= ppg->g_offset_change_x;
   set_center_lon_ppg(ppg);
}
void change_lat_offset( int add ) {
   PPolyGlobs ppg = get_poly_globs();
   if(add)
      ppg->g_offset_lat += ppg->g_offset_change_y;
   else
      ppg->g_offset_lat -= ppg->g_offset_change_y;
   set_center_lat_ppg(ppg);
}

void increase_lon_offset( void ) { change_lon_offset(1); }
void increase_lat_offset( void ) { change_lat_offset(1); }
void decrease_lon_offset( void ) { change_lon_offset(0); }
void decrease_lat_offset( void ) { change_lat_offset(0); }

void reset_lat_lon_offset( void )
{
   PPolyGlobs ppg = get_poly_globs();
   ppg->g_offset_lat = 0.0;
   ppg->g_offset_lon = 0.0;
   set_center_lat_ppg(ppg);
   set_center_lon_ppg(ppg);
}

static void set_g_center_lon( double lon )
{
   PPolyGlobs ppg = get_poly_globs();
   ppg->g_prev_c_lon = ppg->g_set_c_lon;
   ppg->g_set_c_lon = lon;
}
static void set_g_center_lat( double lat )
{
   PPolyGlobs ppg = get_poly_globs();
   ppg->g_prev_c_lat = ppg->g_set_c_lat;
   ppg->g_set_c_lat = lat;
}
static int got_prev_center = 0;
void set_g_center_lon_lat( double lon, double lat )
{
   set_g_center_lon( lon );
   set_g_center_lat( lat );
   reset_lat_lon_offset();
   calculate_scroll_sizes();
   got_prev_center = 1;
}

int get_g_user_center(void) { return get_poly_globs()->g_user_center; }

void set_user_center_lon_lat( double lon, double lat )
{
    set_g_center_lon( lon );
    set_g_center_lat( lat );
    reset_lat_lon_offset();
    get_poly_globs()->g_user_center = 1;    // get_g_user_center();
}   

void Restore_Previous( void )   
{
   PPolyGlobs ppg = get_poly_globs();
   if(got_prev_zoom)
      set_g_map_zoom(ppg->g_prev_map_zoom);
   else
      set_g_map_zoom(DEFAULT_ZOOM);

   if(got_prev_center)
      set_g_center_lon_lat( ppg->g_prev_c_lon, ppg->g_prev_c_lat );

   calculate_scroll_sizes();
}

TGPolyLoad * get_new_loaded_poly( void )
{
   TGPolyLoad * ppl = new TGPolyLoad;
   loaded_polys.push_back(ppl);
   enabled_polys.push_back(1);   // equivalent list - default to enabled
   sprtf("Set new 'active_polyload' to %p\n", ppl );
   active_polyload = ppl;
   return ppl;
}


TGPolyLoad * get_next_loaded_poly( void )
{
   TGPolyLoad * ppl = NULL;
   if( next_loaded_poly < loaded_polys.size() ) {
      ppl = loaded_polys[next_loaded_poly];
      current_loaded_poly = next_loaded_poly;
      next_loaded_poly++;
      active_polyload = ppl;
   }
   sprtf("Get next 'active_polyload' to %p\n", ppl );
   return ppl;
}

TGPolyLoad * get_first_loaded_poly( void )
{
   next_loaded_poly = 0;
   return get_next_loaded_poly();
}


void delete_loaded_poly( void )
{
   int   i;
   for( i = 0; i < loaded_polys.size(); i++ ) {
      TGPolyLoad * pp = loaded_polys[i];
      if(pp->ppc) // if poly counts,
         delete pp->ppc;   // delete poly counts
      pp->ppc = NULL;
      delete pp;
   }
   loaded_polys.clear();
   enabled_polys.clear();
}

TGPolyLoad * get_active_poly( void ) { return active_polyload; }

int is_poly_enabled( int num )
{
   if(( enabled_polys.size() > 0 ) &&
      ( num >= 0) &&
      ( num < enabled_polys.size()))
      return ( enabled_polys[num] );
   return 0;
}

int get_poly_enabled_count(void)
{
   int   i, cnt;
   cnt = 0;
   for( i = 0; i < enabled_polys.size(); i++ ) {
      if( enabled_polys[i] )
         cnt++;
   }
   return cnt;
}

int is_current_poly_enabled( void )
{
   return is_poly_enabled(current_loaded_poly);
}

void toggle_poly_enabled( int num )
{
   if(( enabled_polys.size() > 0 ) &&
      ( num >= 0) &&
      ( num < enabled_polys.size())) {
      if( enabled_polys[num] )
         enabled_polys[num] = 0;
      else
         enabled_polys[num] = 1;
   }
}

// ==========================================================

typedef struct tagGetPoint {
   int   area_type;
   poly_list pl;
   poly_list_iterator pi;
   TGPolygon tgpg;
   int cont;
   point_list contour;
   point_list::iterator pli;
   Point3D p;
   int poly_number, contour_number;
}GetPoint;

GetPoint * curr_pgp = NULL;

void * get_first_point( PsGETPT psgp )
{
   double * px = psgp->px;
   double * py = psgp->py;
   int * pcnum = psgp->pcnum;
   TGPolyLoad * ppl = get_active_poly();
   if( !ppl )
      return NULL;
   if( curr_pgp )
      delete curr_pgp;
   GetPoint *pgp = new GetPoint;
   curr_pgp = pgp;
   pgp->area_type = 0;
   pgp->poly_number = 0;
   pgp->contour_number = 0;
   while(pgp->area_type < TG_MAX_AREA_TYPES) {
      pgp->pl = ppl->polys_in.polys[pgp->area_type];
      if( pgp->pl.size() ) {
         pgp->poly_number++;
         for( pgp->pi = pgp->pl.begin(); pgp->pi != pgp->pl.end(); pgp->pi++ ) {
            pgp->tgpg = *pgp->pi;
            for( pgp->cont = 0; pgp->cont < pgp->tgpg.contours(); pgp->cont++ ) {
               pgp->contour = pgp->tgpg.get_contour(pgp->cont);
               pgp->contour_number++; // new contour
               for( pgp->pli = pgp->contour.begin(); pgp->pli < pgp->contour.end(); pgp->pli++ ) {
                  pgp->p = *pgp->pli;
                  *px = pgp->p.lon();
                  *py = pgp->p.lat();
                  *pcnum = pgp->contour_number; 
                  return pgp;
               }
            }
         }
      }
      pgp->area_type++;
   }
   return NULL;
}

void * get_next_point( void * vp, PsGETPT psgp )
{
   double * px = psgp->px;
   double * py = psgp->py;
   int * pcnum = psgp->pcnum;
   TGPolyLoad * ppl = get_active_poly();
   if( !ppl )
      return NULL;
   GetPoint *pgp = (GetPoint *)vp;
   if (pgp) {
      while(pgp->area_type < TG_MAX_AREA_TYPES) {
         pgp->pli++;
         if ( pgp->pli < pgp->contour.end() ) {
            // continue in same contour
            pgp->p = *pgp->pli;
            *px = pgp->p.lon();
            *py = pgp->p.lat();
            *pcnum = pgp->contour_number; 
            return pgp;
         }
         pgp->cont++;
         if( pgp->cont < pgp->tgpg.contours() ) {
               pgp->contour_number++;  // new contour
               pgp->contour = pgp->tgpg.get_contour(pgp->cont);
               for( pgp->pli = pgp->contour.begin(); pgp->pli < pgp->contour.end(); pgp->pli++ ) {
                  pgp->p = *pgp->pli;
                  *px = pgp->p.lon();
                  *py = pgp->p.lat();
                  *pcnum = pgp->contour_number; 
                  return pgp;
               }
         }
         pgp->pi++;
         if( pgp->pi != pgp->pl.end() ) {
            pgp->tgpg = *pgp->pi;
            for( pgp->cont = 0; pgp->cont < pgp->tgpg.contours(); pgp->cont++ ) {
               pgp->contour = pgp->tgpg.get_contour(pgp->cont);
               pgp->contour_number++;  // new contour
               for( pgp->pli = pgp->contour.begin(); pgp->pli < pgp->contour.end(); pgp->pli++ ) {
                  pgp->p = *pgp->pli;
                  *px = pgp->p.lon();
                  *py = pgp->p.lat();
                  *pcnum = pgp->contour_number; 
                  return pgp;
               }
            }
         }
         pgp->area_type++; // bump AREA TYPE
         // and look for NEXT poly_list, if there is one
         while(pgp->area_type < TG_MAX_AREA_TYPES) {
            pgp->pl = ppl->polys_in.polys[pgp->area_type];
            if( pgp->pl.size() ) {
               for( pgp->pi = pgp->pl.begin(); pgp->pi != pgp->pl.end(); pgp->pi++ ) {
                  pgp->tgpg = *pgp->pi;
                  for( pgp->cont = 0; pgp->cont < pgp->tgpg.contours(); pgp->cont++ ) {
                     pgp->contour = pgp->tgpg.get_contour(pgp->cont);
                     pgp->contour_number++;  // new contour
                     for( pgp->pli = pgp->contour.begin(); pgp->pli < pgp->contour.end(); pgp->pli++ ) {
                        pgp->p = *pgp->pli;
                        *px = pgp->p.lon();
                        *py = pgp->p.lat();
                        *pcnum = pgp->contour_number; 
                        return pgp; // success for NEXT, if any
                     }
                  }
               }
            }
            pgp->area_type++; // bump AREA TYPE
         }  // while less than MAX
      }
   }
   return NULL;
}

void * get_first_point_BAD( PsGETPT psgp )
{
   TGPolyLoad * ppl = get_active_poly();
   if( !ppl )
      return NULL;
   if( curr_pgp )
      delete curr_pgp;
   GetPoint *pgp = new GetPoint;
   curr_pgp = pgp;
   pgp->area_type = 0;
   pgp->poly_number = 0;
   pgp->contour_number = 0;
   pgp->pl = ppl->polys_in.polys[pgp->area_type];   // set FIRST area type
   return get_next_point( pgp, psgp );
}

PPolyCounter Get_NEW_Counter( void )
{
   int   i;
   PPolyCounter ppc = new PolyCounter;
   ppc->total = 0;
   ppc->points = 0;
   ppc->pmax.setx( -2000.0 );
   ppc->pmax.sety( -2000.0 );
   ppc->pmin.setx( 2000.0 );
   ppc->pmin.sety( 2000.0 );
   for ( i = 0; i < TG_MAX_AREA_TYPES; ++i ) {
      ppc->size[i] = 0;
   }
   return ppc;
}

void set_min_max_spread( PPolyCounter ppc )
{
   double min_lat, max_lat, min_lon, max_lon;
   double c_lat, c_lon, xdist, ydist;
   min_lat = ppc->pmin.lat();
   min_lon = ppc->pmin.lon();
   max_lat = ppc->pmax.lat();
   max_lon = ppc->pmax.lon();
   c_lat = (min_lat + max_lat) / 2.0;
   c_lon = (min_lon + max_lon) / 2.0;
   xdist = max_lon - c_lon;   // hozontal x 1/2 spread
   ydist = max_lat - c_lat;   // vertical y 1/2 spread
   ppc->pcenter.setlon(c_lon);
   ppc->pcenter.setlat(c_lat);
   ppc->pdist.setx(xdist);
   ppc->pdist.sety(ydist);
}

PPolyCounter Set_Poly_Counts( TGPolyLoad * ppl )
{
   if( !ppl )
      return NULL;
   PPolyCounter ppc = ppl->ppc;  // extract counts
   if ( ppc )  // if already done counts,
      return ppc; // then all done
   sGETPT sgp;
   double x, y;
   int   i, cnum;

   sgp.pcnum = &cnum;
   sgp.px    = &x;
   sgp.py    = &y;
   ppc = Get_NEW_Counter();
   GetPoint *pgp = (GetPoint *)get_first_point( &sgp );
   if( pgp ) {
      i = pgp->area_type;
      ppc->size[i] = pgp->pl.size();
      ppc->total += ppc->size[i];
   }
   while( pgp ) {
      if ( x < ppc->pmin.x() )
         ppc->pmin.setx(x);
      if ( x > ppc->pmax.x() )
         ppc->pmax.setx(x);
      if ( y < ppc->pmin.y() )
         ppc->pmin.sety(y);
      if ( y > ppc->pmax.y() )
         ppc->pmax.sety(y);
      ppc->points++;
      pgp = (GetPoint *)get_next_point( pgp, &sgp );
      if( pgp ) {
         if( i != pgp->area_type ) { // if a different area type
            i = pgp->area_type;  // update are type, and get SIZE
            ppc->size[i] = pgp->pl.size();
            ppc->total += ppc->size[i];
         }
      }
   }

   set_min_max_spread( ppc ); // get center, spread, etc...

   ppl->ppc = ppc;   // store counts

   return ppc;
}

PPolyCounter Get_Poly_Counts( void )
{
   TGPolyLoad * ppl = get_active_poly();  // get active poly
   if( !ppl )
      return NULL;
   
   PPolyCounter ppc = ppl->ppc;  // extract counts
   if ( ppc )  // if already done counts,
      return ppc; // then all done

   return Set_Poly_Counts( ppl );
}

void center_on_this( int left, int top, int right, int bottom )
{
   double tllat, tllon, brlat, brlon;
   mm_pt2ll( &tllat, &tllon, left,  top    );
   mm_pt2ll( &brlat, &brlon, right, bottom );
   double c_lat = (tllat + brlat) / 2.0;
   double c_lon = (tllon + brlon) / 2.0;
   // got an area - find the ZOOM
   double min_lat = tllat;
   double min_lon = tllon;
   double max_lat = brlat;
   double max_lon = brlon;
//   double xdist = max_lon > min_lon ? max_lon - min_lon : min_lon - max_lon;
//   double ydist = max_lat > min_lat ? max_lat - min_lat : min_lat - max_lat;
   double xdist = max_lon - min_lon;
   double ydist = max_lat - min_lat;
   double tzoom[5];
   double mapsize = mm_map_size;
   double min_dist = SG_EPSILON; // * 10.0;
#if 0
   if( max_lon > min_lon ) {
      xdist = max_lon - min_lon;
   } else {
      xdist = min_lon;
      min_lon = max_lon;
      max_lon = min_lon;
      xdist = max_lon - min_lon;
   }
#endif
   sprtf("center: lat,lon min %0.6f,%0.6f max %0.6f,%0.6f, spread %0.6f,%0.6f\n",
      min_lat, min_lon,
      max_lat, max_lon,
      xdist, ydist );
   // convert lat,lon = x,y to ZOOM
   if(( abs(xdist) < min_dist )||( abs(ydist) < min_dist )) {
      sprtf("center: only change the center, not the zoom (%g,%g)\n", xdist, ydist );
   } else {
      win_llpt2zoom( min_lat, min_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[0],
           14.0, 28.0 );
           //0.0, 0.0 );
      win_llpt2zoom( max_lat, max_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[1],
           mapsize - 14.0, mapsize - 28.0);
           //mapsize, mapsize );
      win_llpt2zoom( min_lat, max_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[2],
           14.0, mapsize - 28.0 );
           //0.0, mapsize );
      win_llpt2zoom( max_lat, min_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[3],
           mapsize - 14.0, 28.0 );
           //mapsize, 0.0 );
      tzoom[4] = tzoom[0];
      if ( tzoom[1] > tzoom[4] )
         tzoom[4] = tzoom[1];
      if ( tzoom[2] > tzoom[4] )
         tzoom[4] = tzoom[2];
      if ( tzoom[3] > tzoom[4] )
         tzoom[4] = tzoom[3];
      sprtf( "center: New ZOOM %g, range TL=%g, BR=%g, BL=%g, TR=%g Was %g\n",
         tzoom[4], tzoom[0], tzoom[1], tzoom[2], tzoom[3], mm_map_zoom );
      set_g_map_zoom(tzoom[4]); // SET the CALCULATED ZOOM factor
   }
   sprtf("center: lat,lon min %0.6f,%0.6f\n", c_lat, c_lon );
   set_g_center_lon_lat( c_lon, c_lat );  // SET new CENTER - with any offset reset
   // need to repaint to see results...
}

void write_out_zoom(void)
{
   double map_zoom = get_g_map_zoom();
   sprtf("Current Zoom: %G (%0.6fM) (%0.6fMM)\n", map_zoom,
      (map_zoom * 1000), (map_zoom * 1000000));
}

void write_out_center(void)
{
   double map_zoom = get_g_map_zoom();
   double lon = get_g_center_lon();
   double lat = get_g_center_lat();
   sprtf("--center-lon-lat=%.12f,%.12f --zoom=%.6f\n",
       lon, lat, map_zoom * 1000);
}

// eof - poly-load.cxx
