// poly-bmp.cxx

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <windows.h>
#include "lib_sprtf.hxx"
#include "poly-bmp.hxx"

static BITMAP  bmp;
static WORD    cClrBits;
static SIZE_T  uBytes;
static SIZE_T  uImageSize;
static int ImgWidth, ImgHeight;

// PBITMAPINFO CreateBitmapInfoStruct(HWND hwnd, HBITMAP hBmp)
PBITMAPINFO CreateBitmapInfoStruct(HBITMAP hBmp, int * pwid = NULL, int * phgt = NULL)
{
   PBITMAPINFO pbmi; 
   // Retrieve the bitmap color format, width, and height.
   if ( !GetObject(hBmp, sizeof(BITMAP), (LPSTR)&bmp) ) {
      sprtf( "ERROR: GetObject FAILED!\n" );
      return NULL;
   }
   ImgWidth  = bmp.bmWidth;
   ImgHeight = bmp.bmHeight;
   if( pwid )
      *pwid = ImgWidth;
   if( phgt )
      *phgt = ImgHeight;

   // Convert the color format to a count of bits.  
   cClrBits = (WORD)(bmp.bmPlanes * bmp.bmBitsPixel); 
   if (cClrBits <= 1)
      cClrBits = 1;
   else if (cClrBits <= 4)
      cClrBits = 4;
   else if (cClrBits <= 8)
      cClrBits = 8;
   else if (cClrBits <= 16)
      cClrBits = 16;
   else if (cClrBits <= 24)
      cClrBits = 24;
   else cClrBits = 32; 

   // Allocate memory for the BITMAPINFO structure. (This structure  
   // contains a BITMAPINFOHEADER structure and an array of RGBQUAD 
   // data structures.)  

   if (cClrBits != 24)
      uBytes = (sizeof(BITMAPINFOHEADER) + 
               sizeof(RGBQUAD) * (1<< cClrBits)); 
     // There is no RGBQUAD array for the 24-bit-per-pixel format.  
   else
      uBytes = sizeof(BITMAPINFOHEADER); 

   pbmi = (PBITMAPINFO) LocalAlloc(LPTR, uBytes );
   if ( !pbmi ) {
      sprtf( "ERROR: LocalAlloc FAILED!\n" );
      return NULL;
   }
   // Initialize the fields in the BITMAPINFO structure.  
   pbmi->bmiHeader.biSize = sizeof(BITMAPINFOHEADER); 
   pbmi->bmiHeader.biWidth = bmp.bmWidth; 
   pbmi->bmiHeader.biHeight = bmp.bmHeight; 
   pbmi->bmiHeader.biPlanes = bmp.bmPlanes; 
   pbmi->bmiHeader.biBitCount = bmp.bmBitsPixel; 
   if (cClrBits < 24)
      pbmi->bmiHeader.biClrUsed = (1<<cClrBits); 

   // If the bitmap is not compressed, set the BI_RGB flag.
   pbmi->bmiHeader.biCompression = BI_RGB; 

   // Compute the number of bytes in the array of color  
   // indices and store the result in biSizeImage.  
   // For Windows NT, the width must be DWORD aligned unless  
   // the bitmap is RLE compressed. This example shows this.  
   // For Windows 95/98/Me, the width must be WORD aligned unless the  
   // bitmap is RLE compressed. 
   uImageSize = ((pbmi->bmiHeader.biWidth * cClrBits +31) & ~31) / 8
      * pbmi->bmiHeader.biHeight; 
   pbmi->bmiHeader.biSizeImage = (DWORD)uImageSize;

   // Set biClrImportant to 0, indicating that all of the  
   // device colors are important.  
   pbmi->bmiHeader.biClrImportant = 0;
   
   return pbmi;
}

//int CreateBMPFile(HWND hwnd, LPTSTR pszFile, PBITMAPINFO pbi, 
//                  HBITMAP hBMP, HDC hDC) 
int CreateBMPFile(LPTSTR pszFile, PBITMAPINFO pbi, 
                  HBITMAP hBMP, HDC hDC) 
{
   HANDLE hf;                 // file handle  
   BITMAPFILEHEADER hdr;       // bitmap file-header  
   PBITMAPINFOHEADER pbih;     // bitmap info-header  
   LPBYTE lpBits;              // memory pointer  
   DWORD dwTotal;              // total count of bytes  
   DWORD cb;                   // incremental count of bytes  
   BYTE *hp;                   // byte pointer  
   DWORD dwTmp; 

   pbih = (PBITMAPINFOHEADER) pbi; 
   lpBits = (LPBYTE) GlobalAlloc(GMEM_FIXED, pbih->biSizeImage);
   if (!lpBits) {
      sprtf( "ERROR: GlobalAlloc(GMEM_FIXED,%d) FAILED!\n", pbih->biSizeImage );
      return 0; // errhandler("GlobalAlloc", hwnd); 
   }
   // Retrieve the color table (RGBQUAD array) and the bits  
   // (array of palette indices) from the DIB.  
   if ( !GetDIBits(hDC, hBMP, 0, (WORD) pbih->biHeight, lpBits, pbi, 
        DIB_RGB_COLORS))
   {
      sprtf( "ERROR: GetDIBits FAILED!\n" );
      GlobalFree((HGLOBAL)lpBits);
      return 0; // errhandler("GetDIBits", hwnd);
   }

   // Create the .BMP file.  
   hf = CreateFile(pszFile, 
                   GENERIC_READ | GENERIC_WRITE, 
                   (DWORD) 0, 
                    NULL, 
                   CREATE_ALWAYS, 
                   FILE_ATTRIBUTE_NORMAL, 
                   (HANDLE) NULL); 
   if (hf == INVALID_HANDLE_VALUE) {
      sprtf( "ERROR: CreateFile FAILED!\n" );
      GlobalFree((HGLOBAL)lpBits);
      return 0; // errhandler("CreateFile", hwnd); 
   }

   hdr.bfType = 0x4d42;        // 0x42 = "B" 0x4d = "M"  
   // Compute the size of the entire file.  
   hdr.bfSize = (DWORD) (sizeof(BITMAPFILEHEADER) + 
                 pbih->biSize + pbih->biClrUsed 
                 * sizeof(RGBQUAD) + pbih->biSizeImage); 
   hdr.bfReserved1 = 0; 
   hdr.bfReserved2 = 0; 

   // Compute the offset to the array of color indices.  
   hdr.bfOffBits = (DWORD) sizeof(BITMAPFILEHEADER) + 
                    pbih->biSize + pbih->biClrUsed 
                    * sizeof (RGBQUAD); 
   // if one used, else to the bits...

   // Copy the BITMAPFILEHEADER into the .BMP file.  
   if ( !WriteFile(hf, (LPVOID) &hdr, sizeof(BITMAPFILEHEADER), 
        (LPDWORD) &dwTmp,  NULL)) 
   {
      sprtf( "ERROR: WriteFile FAILED!\n" );
      GlobalFree((HGLOBAL)lpBits);
      return 0; // errhandler("WriteFile", hwnd); 
   }

   // Copy the BITMAPINFOHEADER and RGBQUAD array into the file.  
   if ( !WriteFile(hf, (LPVOID) pbih, sizeof(BITMAPINFOHEADER) 
                  + pbih->biClrUsed * sizeof (RGBQUAD), 
                  (LPDWORD) &dwTmp, ( NULL))) {
      sprtf( "ERROR: WriteFile FAILED!\n" );
      GlobalFree((HGLOBAL)lpBits);
      return 0; // errhandler("WriteFile", hwnd); 
   }
   // Copy the array of color indices into the .BMP file.  
   dwTotal = cb = pbih->biSizeImage; 
   hp = lpBits; 
   if ( !WriteFile(hf, (LPSTR) hp, (int) cb, (LPDWORD) &dwTmp,NULL)) {
      sprtf( "ERROR: WriteFile FAILED!\n" );
      GlobalFree((HGLOBAL)lpBits);
      return 0; // errhandler("WriteFile", hwnd); 
   }
   // Close the .BMP file.  
   if ( !CloseHandle(hf) ) {
      sprtf( "ERROR: CloseHandle FAILED!\n" );
      GlobalFree((HGLOBAL)lpBits);
      return 0; // errhandler("CloseHandle", hwnd); 
   }

   // Free memory.  
   GlobalFree((HGLOBAL)lpBits);
   sprtf("Written %s: w=%d, h=%d\n", pszFile, ImgWidth, ImgHeight );
   return 1;   // SUCCESS
}

int write_bmp_file( HDC hdc, HBITMAP hBmp, char * filename )
{
   PBITMAPINFO pbmi = CreateBitmapInfoStruct(hBmp);
   if( !pbmi )
      return 0;

   if( !CreateBMPFile(filename, pbmi, hBmp, hdc ) )
      return 0;

   LocalFree(pbmi);

   return 1;
}

LPBYTE get_bitmap_bits(PBITMAPINFO pbi, HBITMAP hBMP, HDC hDC) 
{
   PBITMAPINFOHEADER pbih;     // bitmap info-header  
   LPBYTE lpBits;              // memory pointer  

   pbih = (PBITMAPINFOHEADER) pbi;
   if ( !pbih->biSizeImage )
      return NULL;

   lpBits = (LPBYTE) GlobalAlloc(GMEM_FIXED, pbih->biSizeImage);

   if (!lpBits)
      return NULL; // errhandler("GlobalAlloc", hwnd); 

   // Retrieve the color table (RGBQUAD array) and the bits  
   // (array of palette indices) from the DIB.  
   if ( !GetDIBits(hDC, hBMP, 0, (WORD) pbih->biHeight, lpBits, pbi, 
        DIB_RGB_COLORS))
   {
      GlobalFree(lpBits);
      return NULL; // errhandler("GetDIBits", hwnd);
   }
   return lpBits;
}

unsigned char * get_image_bits( HDC hdc, HBITMAP hBmp, int * psize )
{
   int width, height;
   PBITMAPINFO pbmi = CreateBitmapInfoStruct(hBmp, &width, &height);
   if( !pbmi )
      return NULL;

   if( !(( cClrBits == 24 )||( cClrBits == 32 )) ) {
      sprtf( "Presently only coded to deal with 24 or 32-BBP bitmaps. Got %d\n", cClrBits );
      LocalFree(pbmi);
      return NULL;
   }

   LPBYTE lpBits = get_bitmap_bits(pbmi, hBmp, hdc);  // memory pointer
   if( !lpBits ) {
      LocalFree(pbmi);
      return NULL;
   }
   int current_size = width;
   if(height > width)
      current_size = height;
   *psize = current_size;
   int size = (current_size * current_size * 3);
   int brow = ((pbmi->bmiHeader.biWidth * cClrBits +31) & ~31) / 8;
   unsigned char * image = (unsigned char *)malloc(size);
   if( !image ) {
      LocalFree(pbmi);
      GlobalFree(lpBits);
      sprtf( "ERROR: memory allocation of %d bytes FAILED!\n", size );
      return NULL;
   }
   // ready to party with the bits...
   int x, y, w, h, irow, y3, w3;
   unsigned char * cp;
   int off = cClrBits / 8;

   irow = current_size * 3;   // one image ROW
   w = h = 0;
   //for( x = height - 1; x >= 0; x-- ) {   // start with to TOP row (at bottom of BMP)
   for( x = 0; x < height; x++ ) {   // start with BOTTOM row (at top of BMP)
      PBYTE pb = &lpBits[x * brow]; // get pointer to this BMP ROW
      cp = &image[ h * irow ]; // and image row
      for( y = 0; y < width; y++ ) {
         y3 = y * off;   // get offset into row
         w3 = w * 3;
         BYTE r = pb[y3 + 0]; // collect RGB
         BYTE g = pb[y3 + 1];
         BYTE b = pb[y3 + 2];
         cp[w3 + 0] = r;      // set RGB
         cp[w3 + 1] = g;
         cp[w3 + 2] = b;
         w++;  // bump to next width
      }
      while(w < current_size) {
         // fill any remainder, of this row, with gray
         w3 = w * 3;   // get offset into row
         cp[w3 + 0] = 0x90;
         cp[w3 + 0] = 0x90;
         cp[w3 + 0] = 0x90;
         w++;
      }
      // done this entire ROW
      w = 0;
      h++;
   }
   while(h < current_size) {
      cp = &image[ h * irow ]; // and image row
      while(w < current_size) {
         // fill any ramainder with gray
         w3 = w * 3;   // get offset into row
         cp[w3 + 0] = 0x90;
         cp[w3 + 0] = 0x90;
         cp[w3 + 0] = 0x90;
         w++;  // bump offset into row
      }
      h++;  // bump row
   }

   return image;
}

unsigned char * get_image_bits2( HDC hdc, HBITMAP hBmp, int * pwidth, int * pheight )
{
   int width, height;
   PBITMAPINFO pbmi = CreateBitmapInfoStruct(hBmp, &width, &height);
   if( !pbmi )
      return NULL;

   if( !(( cClrBits == 24 )||( cClrBits == 32 )) ) {
      sprtf( "Presently only coded to deal with 24 or 32-BBP bitmaps. Got %d\n", cClrBits );
      LocalFree(pbmi);
      return NULL;
   }

   LPBYTE lpBits = get_bitmap_bits(pbmi, hBmp, hdc);  // memory pointer
   if( !lpBits ) {
      LocalFree(pbmi);
      return NULL;
   }
   *pwidth = width;
   *pheight = height;
   int size = (width * height * 3);
   int brow = ((pbmi->bmiHeader.biWidth * cClrBits +31) & ~31) / 8;
   unsigned char * image = (unsigned char *)malloc(size);
   if( !image ) {
      LocalFree(pbmi);
      GlobalFree(lpBits);
      sprtf( "ERROR: memory allocation of %d bytes FAILED!\n", size );
      return NULL;
   }
   // ready to party with the bits...
   int x, y, w, h, irow, y3, w3;
   unsigned char * cp;
   int off = cClrBits / 8;

   irow = width * 3;   // one image ROW
   w = h = 0;
   //for( x = height - 1; x >= 0; x-- ) {   // start with to TOP row (at bottom of BMP)
   for( x = 0; x < height; x++ ) {   // start with BOTTOM row (at top of BMP)
      PBYTE pb = &lpBits[x * brow]; // get pointer to this BMP ROW
      cp = &image[ h * irow ]; // and image row
      for( y = 0; y < width; y++ ) {
         y3 = y * off;   // get offset into row
         w3 = w * 3;
         BYTE r = pb[y3 + 0]; // collect RGB
         BYTE g = pb[y3 + 1];
         BYTE b = pb[y3 + 2];
         cp[w3 + 0] = r;      // set RGB
         cp[w3 + 1] = g;
         cp[w3 + 2] = b;
         w++;  // bump to next width
      }
      // done this entire ROW
      w = 0;
      h++;
   }

   return image;
}

// eof - poly-bmp.cxx
