// poly-array.cxx

// array.cxx -- Array management class
//
// Written by Curtis Olson, started March 1998.
//
// Copyright (C) 1998 - 1999  Curtis L. Olson  - http://www.flightgear.org/~curt
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// $Id: array.cxx,v 1.24 2005-11-10 16:26:59 curt Exp $


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <simgear/compiler.h>

#include <iostream>
#include <string>

#include <simgear/constants.h>
#include <simgear/misc/sgstream.hxx>
#include <simgear/misc/strutils.hxx>
#include <simgear/misc/sg_path.hxx>
#include <simgear/math/leastsqs.hxx>

#include "poly-array.hxx"
#include "lib_sprtf.hxx"

using std::string;
using std::cout;
using std::endl;

#define m_printf sprtf

static int dbg_check(void)
{
    int i;
    i = 1;
    return i;
}


PVArray::PVArray( void ):
  array_in(NULL),
  fitted_in(NULL),
  fitted_size(0)
{
    sprtf("class PVArray constructor (void) called.\n");
    //in_data = new int[ARRAY_SIZE_1][ARRAY_SIZE_1];
    // out_data = new float[ARRAY_SIZE_1][ARRAY_SIZE_1];
    allocate_mem();
}


PVArray::PVArray( const string &file ):
  array_in(NULL),
  fitted_in(NULL),
  fitted_size(0)
{
    sprtf("class PVArray constructor (file) called.\n");
    //in_data = new int[ARRAY_SIZE_1][ARRAY_SIZE_1];
    // out_data = new float[ARRAY_SIZE_1][ARRAY_SIZE_1];
    allocate_mem();

    PVArray::open(file);
}


// open an Array file (and fitted file if it exists)
bool PVArray::open( const string& file_base ) {
    bool success = true;

    // open array data file
    string array_name = file_base + ".arr.gz";
    array_in = new sg_gzifstream( array_name );
    if ( ! array_in->is_open() ) {
        sprtf("ERROR: Cannot open [%s] file!\n", array_name.c_str() );
        success = false;
        return success;
    } else {
        sprtf("Opening array data file: [%s]\n", array_name.c_str() );
    }
    in_array = array_name;
    // open fitted data file
    string fitted_name = file_base + ".fit.gz";
    fitted_in = new sg_gzifstream( fitted_name );
    if ( ! fitted_in->is_open() ) {
        // not having a .fit file is unfortunate, but not fatal.  We
        // can do a really stupid/crude fit on the fly, but it will
        // not be nearly as nice as what the offline terrafit utility
        // would have produced.
        sprtf("WARNING: Cannot open [%s] file!\n", fitted_name.c_str() );
    } else {
        sprtf("Opening fitted data file: [%s]\n", fitted_name.c_str() );
        in_fitted = fitted_name;
    }

    return success;
}


// close an Array file
bool
PVArray::close() {
    // the sg_gzifstream doesn't seem to have a close()

    if (array_in)
        delete array_in;
    if (fitted_in)
        delete fitted_in;
    array_in = NULL;
    fitted_in = NULL;
    return true;
}


// parse Array file, pass in the bucket so we can make up values when
// the file wasn't found.
int
PVArray::parse( SGBucket& b ) {

    int iret = 0;
    int i, j, height;
    // Parse/load the array data file
    if ( array_in->is_open() ) {
	sprtf("File %s open, parse...\n", in_array.c_str() );

        *array_in >> originx >> originy;
	*array_in >> cols >> col_step;
	*array_in >> rows >> row_step;

        min_lon = originx / 3600.0;
        min_lat = originy / 3600.0;
        max_lon = (originx + (cols * col_step)) / 3600.0;
        max_lat = (originy + (rows * row_step)) / 3600.0;

	sprtf("    origin (x,y) = %g,%g\n", originx, originy );
	sprtf("    cols = %d, rows = %d\n", cols, rows );
	sprtf("    col_step = %f, row_step = %f\n", col_step, row_step );
	sprtf("    range (lon,lat) = %g,%g to %g,%g\n", min_lon, min_lat, max_lon, max_lat );
        max_height = -999999;
        min_height =  999999;
	for ( i = 0; i < cols; i++ ) {
	    for ( j = 0; j < rows; j++ ) {
		*array_in >> height;
                in_data[i][j] = height;
                if ( height > max_height ) max_height = height;
                if ( height < min_height ) min_height = height;
	    }
	}
	sprtf("    Done parsing %d points. Min %d, Max %d meters.\n",
            rows * cols,
            min_height, max_height );

        iret |= 1;
    } else {
	sprtf("File not open (not found?), fill with zero'd data...\n" );

	originx = ( b.get_center_lon() - 0.5 * b.get_width() ) * 3600.0;
	originy = ( b.get_center_lat() - 0.5 * b.get_height() ) * 3600.0;

	double max_x = ( b.get_center_lon() + 0.5 * b.get_width() ) * 3600.0;
	double max_y = ( b.get_center_lat() + 0.5 * b.get_height() ) * 3600.0;

	cols = 3;
	col_step = (max_x - originx) / (cols - 1);
	rows = 3;
	row_step = (max_y - originy) / (rows - 1);

	sprtf("    origin  = %g,%g (x,y)\n", originx, originy );
	sprtf("    cols = %d, rows = %d\n", cols, rows );
	sprtf("    col_step = %f, row_step = %f\n", col_step, row_step );

        max_height = 0;
        min_height = 0;
	for ( i = 0; i < cols; i++ ) {
	    for ( j = 0; j < rows; j++ ) {
		in_data[i][j] = 0;
	    }
	}

        sprtf( "    Zero fill done...\n" );
    }

    // Parse/load the fitted data file
    if ( fitted_in->is_open() ) {
        double x, y, z;
        *fitted_in >> fitted_size;
        sprtf("Loading the fitted data - %d points...\n", fitted_size);
        for ( int i = 0; i < fitted_size; ++i ) {
            *fitted_in >> x >> y >> z;
            fitted_list.push_back( Point3D(x, y, z) );
            // cout << " loading fitted = " << Point3D(x, y, z) << endl;
        }
        fit_to_array_grid();
        iret |= 2;
    }

    return iret;
}


// write an Array file
bool PVArray::write( const string root_dir, SGBucket& b ) {
    // generate output file name
    string base = b.gen_base_path();
    string path = root_dir + "/" + base;
    SGPath sgp( path );
    sgp.append( "dummy" );
    sgp.create_dir( 0755 );

    string array_file = path + "/" + b.gen_index_str() + ".arr.new.gz";
    sprtf("Out array_file = %s...\n", array_file.c_str() );

    // write the file
    gzFile fp;
    if ( (fp = gzopen( array_file.c_str(), "wb9" )) == NULL ) {
        sprtf("ERROR:  cannot open/create [%s] for writing!\n", array_file.c_str() );
	return false;
    }

    sprtf("origin x=%f, y=%f\n", originx, originy );
    gzprintf( fp, "%d %d\n", (int)originx, (int)originy );
    //gzprintf( fp, "%d %d %d %d\n", cols, (int)col_step, rows, (int)row_step );
    gzprintf( fp, "%d %f %d %f\n", cols, col_step, rows, row_step ); // allow for say 0.75 steps
    for ( int i = 0; i < cols; ++i ) {
	for ( int j = 0; j < rows; ++j ) {
	    gzprintf( fp, "%d ", (int)in_data[i][j] );
	}
	gzprintf( fp, "\n" );
    }
    gzclose(fp);

    return true;
}


// do our best to remove voids by picking data from the nearest neighbor.
void PVArray::remove_voids( ) {
    // need two passes to ensure that all voids are removed (unless entire
    // array is a void.)
    bool have_void = true;
    int last_elev = -32768;
    for ( int pass = 0; pass < 2 && have_void; ++pass ) { 
        // attempt to fill in any void data horizontally
        for ( int i = 0; i < cols; i++ ) {
            int j;

            // fill in front ways
            last_elev = -32768;
            have_void = false;
            for ( j = 0; j < rows; j++ ) {
                if ( in_data[i][j] > -9000 ) {
                    last_elev = in_data[i][j];
                } else if ( last_elev > -9000 ) {
                    in_data[i][j] = last_elev;
                } else {
                    have_void = true;
                }
            }
            // fill in back ways
            last_elev = -32768;
            have_void = false;
            for ( j = rows - 1; j >= 0; j-- ) {
                if ( in_data[i][j] > -9000 ) {
                    last_elev = in_data[i][j];
                } else if ( last_elev > -9000 ) {
                    in_data[i][j] = last_elev;
                } else {
                    have_void = true;
                }
            }
        }

        // attempt to fill in any void data vertically
        for ( int j = 0; j < rows; j++ ) {
            int i;

            // fill in front ways
            last_elev = -32768;
            have_void = false;
            for ( i = 0; i < cols; i++ ) {
                if ( in_data[i][j] > -9999 ) {
                    last_elev = in_data[i][j];
                } else if ( last_elev > -9999 ) {
                    in_data[i][j] = last_elev;
                } else {
                    have_void = true;
                }
            }

            // fill in back ways
            last_elev = -32768;
            have_void = false;
            for ( i = cols - 1; i >= 0; i-- ) {
                if ( in_data[i][j] > -9999 ) {
                    last_elev = in_data[i][j];
                } else if ( last_elev > -9999 ) {
                    in_data[i][j] = last_elev;
                } else {
                    have_void = true;
                }
            }
        }
    }

    if ( have_void ) {
        // after all that work we still have a void, likely the
        // entire array is void.  Fill in the void areas with zero
        // as a panic fall back.
        for ( int i = 0; i < cols; i++ ) {
            for ( int j = 0; j < rows; j++ ) {
                if ( in_data[i][j] <= -9999 ) {
                    in_data[i][j] = 0;
                }
            }
        }
    }
}


// add a node to the output corner node list
void PVArray::add_corner_node( int i, int j, double val ) {
    
    double x = (originx + i * col_step) / 3600.0;
    double y = (originy + j * row_step) / 3600.0;
    // cout << "originx = " << originx << "  originy = " << originy << endl;
    sprtf("corner = %g,%g,%g\n", x, y, val );
    corner_list.push_back( Point3D(x, y, val) );
}


// add a node to the output fitted node list
void PVArray::add_fit_node( int i, int j, double val ) {
    double x = (originx + (i * col_step)) / 3600.0;
    double y = (originy + (j * row_step)) / 3600.0;
    // cout << Point3D(x, y, val) << endl;
    fitted_list.push_back( Point3D(x, y, val) );
}


#if 0
// Use least squares to fit a simpler data set to dem data.  Return
// the number of fitted nodes.  This is a horrible approach that
// doesn't really work, but it's better than nothing if you've got
// nothing.  Using src/Prep/ArrayFit to create .fit files from the
// .arr files is a *much* better approach, but it is slower which is
// why it needs to be done "offline".
int PVArray::fit( double error ) {
    if ( ! fit_on_the_fly ) {
        return fitted_list.size();
    }
    double x[ARRAY_SIZE_1], y[ARRAY_SIZE_1];
    double m, b, max_error, error_sq;
    double x1, y1;
    // double ave_error;
    double cury, lasty;
    int n, row, start, end;
    int colmin, colmax, rowmin, rowmax;
    bool good_fit;
    // FILE *dem, *fit, *fit1;

    error_sq = error * error;

    cout << "  Initializing fitted node list" << endl;
    corner_list.clear();
    fitted_list.clear();

    // determine dimensions
    colmin = 0;
    colmax = cols;
    rowmin = 0;
    rowmax = rows;
    cout << "  Fitting region = " << colmin << "," << rowmin << " to " 
	 << colmax << "," << rowmax << endl;;
    
    // generate corners list
    add_corner_node( colmin, rowmin, in_data[colmin][rowmin] );
    add_corner_node( colmin, rowmax-1, in_data[colmin][rowmax] );
    add_corner_node( colmax-1, rowmin, in_data[colmax][rowmin] );
    add_corner_node( colmax-1, rowmax-1, in_data[colmax][rowmax] );

    cout << "  Beginning best fit procedure" << endl;
    lasty = 0;

    for ( row = rowmin; row < rowmax; row++ ) {
	// fit  = fopen("fit.dat",  "w");
	// fit1 = fopen("fit1.dat", "w");

	start = colmin;

	// cout << "    fitting row = " << row << endl;

	while ( start < colmax - 1 ) {
	    end = start + 1;
	    good_fit = true;

	    x[0] = start * col_step;
	    y[0] = in_data[start][row];

	    x[1] = end * col_step;
	    y[1] = in_data[end][row];

	    n = 2;

	    // cout << "Least square of first 2 points" << endl;
	    least_squares(x, y, n, &m, &b);

	    end++;

	    while ( (end < colmax) && good_fit ) {
		++n;
		// cout << "Least square of first " << n << " points" << endl;
		x[n-1] = x1 = end * col_step;
		y[n-1] = y1 = in_data[end][row];
		least_squares_update(x1, y1, &m, &b);
		// ave_error = least_squares_error(x, y, n, m, b);
		max_error = least_squares_max_error(x, y, n, m, b);

		/*
		printf("%d - %d  ave error = %.2f  max error = %.2f  y = %.2f*x + %.2f\n", 
		start, end, ave_error, max_error, m, b);
		
		f = fopen("gnuplot.dat", "w");
		for ( j = 0; j <= end; j++) {
		    fprintf(f, "%.2f %.2f\n", 0.0 + ( j * col_step ), 
			    in_data[row][j]);
		}
		for ( j = start; j <= end; j++) {
		    fprintf(f, "%.2f %.2f\n", 0.0 + ( j * col_step ), 
			    in_data[row][j]);
		}
		fclose(f);

		printf("Please hit return: "); gets(junk);
		*/

		if ( max_error > error_sq ) {
		    good_fit = false;
		}
		
		end++;
	    }

	    if ( !good_fit ) {
		// error exceeded the threshold, back up
		end -= 2;  // back "end" up to the last good enough fit
		n--;       // back "n" up appropriately too
	    } else {
		// we popped out of the above loop while still within
		// the error threshold, so we must be at the end of
		// the data set
		end--;
	    }
	    
	    least_squares(x, y, n, &m, &b);
	    // ave_error = least_squares_error(x, y, n, m, b);
	    max_error = least_squares_max_error(x, y, n, m, b);

	    /*
	    printf("\n");
	    printf("%d - %d  ave error = %.2f  max error = %.2f  y = %.2f*x + %.2f\n", 
		   start, end, ave_error, max_error, m, b);
	    printf("\n");

	    fprintf(fit1, "%.2f %.2f\n", x[0], m * x[0] + b);
	    fprintf(fit1, "%.2f %.2f\n", x[end-start], m * x[end-start] + b);
	    */

	    if ( start > colmin ) {
		// skip this for the first line segment
		cury = m * x[0] + b;
		add_fit_node( start, row, (lasty + cury) / 2 );
		// fprintf(fit, "%.2f %.2f\n", x[0], (lasty + cury) / 2);
	    }

	    lasty = m * x[end-start] + b;
	    start = end;
	}

	/*
	fclose(fit);
	fclose(fit1);

	dem = fopen("gnuplot.dat", "w");
	for ( j = 0; j < ARRAY_SIZE_1; j++) {
	    fprintf(dem, "%.2f %.2f\n", 0.0 + ( j * col_step ), 
		    in_data[j][row]);
	} 
	fclose(dem);
	*/

	// NOTICE, this is for testing only.  This instance of
        // output_nodes should be removed.  It should be called only
        // once at the end once all the nodes have been generated.
	// newmesh_output_nodes(&nm, "mesh.node");
	// printf("Please hit return: "); gets(junk);
    }

    // outputmesh_output_nodes(fg_root, p);

    // return fit nodes + 4 corners
    return fitted_list.size() + 4;
}
#endif


// Return the elevation of the closest non-void grid point to lon, lat
double PVArray::closest_nonvoid_elev( double lon, double lat ) const {
    double mindist = 99999999999.9;
    double minelev = -9999.0;
    Point3D p0( lon, lat, 0.0 );

    for ( int row = 0; row < rows; row++ ) {
        for ( int col = 0; col < cols; col++ ) {
            Point3D p1(originx + col * col_step, originy + row * row_step, 0.0);
            double dist = p0.distance3D( p1 );
            double elev = in_data[col][row];
            if ( dist < mindist && elev > -9000 ) {
                mindist = dist;
                minelev = elev;
                // cout << "dist = " << mindist;
                // cout << "  elev = " << elev << endl;
            }
        }
    }

    if ( minelev > -9999.0 ) {
        return minelev;
    } else {
        return 0.0;
    }
}


// return the current altitude based on grid data.  We should rewrite
// this to interpolate exact values, but for now this is good enough
double PVArray::altitude_from_grid( double lon, double lat ) const {
    // we expect incoming (lon,lat) to be in arcsec for now

    double xlocal, ylocal, dx, dy, zA, zB, elev;
    int x1, x2, x3, y1, y2, y3;
    float z1, z2, z3;
    int xindex, yindex;
 
    /* determine if we are in the lower triangle or the upper triangle 
       ______
       |   /|
       |  / |
       | /  |
       |/   |
       ------

       then calculate our end points
     */

    xlocal = (lon - originx) / col_step;
    ylocal = (lat - originy) / row_step;

    xindex = (int)(xlocal);
    yindex = (int)(ylocal);

    // printf("xindex = %d  yindex = %d\n", xindex, yindex);

    if ( xindex + 1 == cols ) {
	xindex--;
    }

    if ( yindex + 1 == rows ) {
	yindex--;
    }

    if ( (xindex < 0) || (xindex + 1 >= cols) ||
	 (yindex < 0) || (yindex + 1 >= rows) ) {
	sprtf("WARNING: Attempt to interpolate value outside of array!!!\n");
	return -9999;
    }

    dx = xlocal - xindex;
    dy = ylocal - yindex;

    if ( dx > dy ) {
	// lower triangle
	// printf("  Lower triangle\n");

	x1 = xindex; 
	y1 = yindex; 
	z1 = in_data[x1][y1];

	x2 = xindex + 1; 
	y2 = yindex; 
	z2 = in_data[x2][y2];
				  
	x3 = xindex + 1; 
	y3 = yindex + 1; 
	z3 = in_data[x3][y3];

	// printf("  dx = %.2f  dy = %.2f\n", dx, dy);
	// printf("  (x1,y1,z1) = (%d,%d,%d)\n", x1, y1, z1);
	// printf("  (x2,y2,z2) = (%d,%d,%d)\n", x2, y2, z2);
	// printf("  (x3,y3,z3) = (%d,%d,%d)\n", x3, y3, z3);

        if ( z1 < -9000 || z2 < -9000 || z3 < -9000 ) {
            // don't interpolate off a void
            return closest_nonvoid_elev( lon, lat );
        }

	zA = dx * (z2 - z1) + z1;
	zB = dx * (z3 - z1) + z1;
	
	// printf("  zA = %.2f  zB = %.2f\n", zA, zB);

	if ( dx > SG_EPSILON ) {
	    elev = dy * (zB - zA) / dx + zA;
	} else {
	    elev = zA;
	}
    } else {
	// upper triangle
	// printf("  Upper triangle\n");

	x1 = xindex; 
	y1 = yindex; 
	z1 = in_data[x1][y1];

	x2 = xindex; 
	y2 = yindex + 1; 
	z2 = in_data[x2][y2];
				  
	x3 = xindex + 1; 
	y3 = yindex + 1; 
	z3 = in_data[x3][y3];

	// printf("  dx = %.2f  dy = %.2f\n", dx, dy);
	// printf("  (x1,y1,z1) = (%d,%d,%d)\n", x1, y1, z1);
	// printf("  (x2,y2,z2) = (%d,%d,%d)\n", x2, y2, z2);
	// printf("  (x3,y3,z3) = (%d,%d,%d)\n", x3, y3, z3);
 
        if ( z1 < -9000 || z2 < -9000 || z3 < -9000 ) {
            // don't interpolate off a void
            return closest_nonvoid_elev( lon, lat );
        }

	zA = dy * (z2 - z1) + z1;
	zB = dy * (z3 - z1) + z1;
	
	// printf("  zA = %.2f  zB = %.2f\n", zA, zB );
	// printf("  xB - xA = %.2f\n", col_step * dy / row_step);

	if ( dy > SG_EPSILON ) {
	    elev = dx * (zB - zA) / dy    + zA;
	} else {
	    elev = zA;
	}
    }

    return elev;
}


#if 0
// Write out a node file that can be used by the "triangle" program.
// Check for an optional "index.node.ex" file in case there is a .poly
// file to go along with this node file.  Include these nodes first
// since they are referenced by position from the .poly file.
void PVArray::outputmesh_output_nodes( const string& fg_root, SGBucket& p )
{
    double exnodes[MAX_EX_NODES][3];
    struct stat stat_buf;
    string dir, file;
    char exfile[256];
#ifdef WIN32
    char tmp_path[256];
#endif
    string command;
    FILE *fd;
    int colmin, colmax, rowmin, rowmax;
    int i, j, count, excount, result;

    // determine dimensions
    colmin = p.get_x() * ( (cols - 1) / 8);
    colmax = colmin + ( (cols - 1) / 8);
    rowmin = p.get_y() * ( (rows - 1) / 8);
    rowmax = rowmin + ( (rows - 1) / 8);
    cout << "  dumping region = " << colmin << "," << rowmin << " to " <<
	colmax << "," << rowmax << "\n";

    // generate the base directory
    string base_path = p.gen_base_path();
    cout << "  fg_root = " << fg_root << "  Base Path = " << base_path << endl;
    dir = fg_root + "/" + base_path;
    cout << "  Dir = " << dir << endl;
    
    // stat() directory and create if needed
    errno = 0;
    result = stat(dir.c_str(), &stat_buf);
    if ( result != 0 && errno == ENOENT ) {
	cout << "  Creating directory\n";

#ifdef _MSC_VER
	fg_mkdir( dir.cstr() );
#else
	command = "mkdir -p " + dir + "\n";
	system( command.c_str() );
#endif
    } else {
	// assume directory exists
    }

    // get index and generate output file name
    file = dir + "/" + p.gen_index_str() + ".node";

    // get (optional) extra node file name (in case there is matching
    // .poly file.
    exfile = file + ".ex";

    // load extra nodes if they exist
    excount = 0;
    if ( (fd = fopen(exfile, "r")) != NULL ) {
	int junki;
	fscanf(fd, "%d %d %d %d", &excount, &junki, &junki, &junki);

	if ( excount > MAX_EX_NODES - 1 ) {
	    printf("Error, too many 'extra' nodes, increase array size\n");
	    exit(-1);
	} else {
	    printf("    Expecting %d 'extra' nodes\n", excount);
	}

	for ( i = 1; i <= excount; i++ ) {
	    fscanf(fd, "%d %lf %lf %lf\n", &junki, 
		   &exnodes[i][0], &exnodes[i][1], &exnodes[i][2]);
	    printf("(extra) %d %.2f %.2f %.2f\n", 
		    i, exnodes[i][0], exnodes[i][1], exnodes[i][2]);
	}
	fclose(fd);
    }

    printf("Creating node file:  %s\n", file);
    fd = fopen(file, "w");

    // first count regular nodes to generate header
    count = 0;
    for ( j = rowmin; j <= rowmax; j++ ) {
	for ( i = colmin; i <= colmax; i++ ) {
	    if ( out_data[i][j] > -9000.0 ) {
		count++;
	    }
	}
	// printf("    count = %d\n", count);
    }
    fprintf(fd, "%d 2 1 0\n", count + excount);

    // now write out extra node data
    for ( i = 1; i <= excount; i++ ) {
	fprintf(fd, "%d %.2f %.2f %.2f\n", 
		i, exnodes[i][0], exnodes[i][1], exnodes[i][2]);
    }

    // write out actual node data
    count = excount + 1;
    for ( j = rowmin; j <= rowmax; j++ ) {
	for ( i = colmin; i <= colmax; i++ ) {
	    if ( out_data[i][j] > -9000.0 ) {
		fprintf(fd, "%d %.2f %.2f %.2f\n", 
			count++, 
			originx + (double)i * col_step, 
			originy + (double)j * row_step,
			out_data[i][j]);
	    }
	}
	// printf("    count = %d\n", count);
    }

    fclose(fd);
}
#endif

void PVArray::free_mem(void)
{
    //if( fit_grid )
    //    delete [] fit_grid;
    if( fit_str )
        delete [] fit_str;

    //fit_grid = NULL;
    fit_str = NULL;
}

int PVArray::allocate_mem(void)
{
    in_data = new int[ARRAY_SIZE_1][ARRAY_SIZE_1];

    fit_str = NULL;

    fit_str = new pvFGS[ARRAY_SIZE_1][ARRAY_SIZE_1];

    if ((fit_str == NULL)||(in_data == NULL)) {
        m_printf("ERROR: MEMORY ALLOCATION FAILED!\n");
        free_mem();
        exit(1);
        return 1;
    }
    // fill fit structure
    for (int i = 0; i < ARRAY_SIZE_1; i++) {
        for (int j = 0; j < ARRAY_SIZE_1; j++) {
            PpvFGS pfgs = &fit_str[i][j];
            pfgs->elevation = M_NEG_ELEV;
            pfgs->flag      = 0;    // NO flags
            pfgs->used      = 0;
        }
    }

    return 0;
}

void PVArray::add_tri(pvTRIV & t)
{
    PpvFGS    pfgs;
    // ------------------
    fit_tris.push_back(t);  // store it
    // ===============

    if (pvVERB9) sprtf( "%04d:TRI:x:y: % 4d,% 4d % 4d,% 4d % 4d,% 4d\n",
        fit_tris.size(),
        t.p1_x, t.p1_y,
        t.p2_x, t.p2_y,
        t.p3_x, t.p3_y );
    // set TRI flags
    pfgs = &fit_str[t.p1_x][t.p1_y];
    pfgs->flag |= pvFLG_TRI1;
    pfgs->used++;

    pfgs = &fit_str[t.p2_x][t.p2_y];
    pfgs->flag |= pvFLG_TRI2;
    pfgs->used++;

    pfgs = &fit_str[t.p3_x][t.p3_y];
    pfgs->flag |= pvFLG_TRI3;
    pfgs->used++;
    // ------------------
}

#define SET_POINTV(o,a,b) \
    pt[o].x = a; \
    pt[o].y = b; \
    t3.ptv[o].pt = pt[o]; \
    t3.ptv[o].vp = SGVec3d(lon,lat,alt)

#define COPY22ONE \
    t.p1   = t.p2; \
    t.p1_x = t.p2_x; \
    t.p1_y = t.p2_y

#define COPY32TWO \
    t.p2   = t.p3; \
    t.p2_x = t.p3_x; \
    t.p2_y = t.p3_y

void PVArray::extract_tris(void)
{
    PpvFGS    pfgs;
    pvTRIV    t;
    pvTRIV3   t3;
    int cols = get_cols();
    int rows = get_rows();
    double col_step = get_col_step();
    double row_step = get_row_step();
    double ox = get_originx();
    double oy = get_originy();
    double lon,lat,alt;
    int x, y, elev;
    int x2, y2, max;
    int i, fnd;
    unsigned int flag;
    pvPT    pt[3];

    //char * buf = _s_tmpbuf;
    x = 0;
    y = 0;
    while(1) {
        pfgs = &fit_str[x][y];
        elev = pfgs->elevation;
        flag = pfgs->flag;
        while ( !flag ) {
            x++;
            if (x < cols) {
                pfgs = &fit_str[x][y];
                elev = pfgs->elevation;
                flag = pfgs->flag;
            } else {
                // check_me();
                break;  // out of points in this ROW
            }
        }
        if ( x < cols ) {   // got FIRST point of triange
            lon = (ox + (x * col_step)) / 3600.0;
            lat = (oy + (y * row_step)) / 3600.0;
            alt = (double)elev;
            t.p1 = SGVec3d(lon,lat,alt); // got FIRST point of triange
            t.p1_x = x;
            t.p1_y = y;
            SET_POINTV(0,x,y);
            x2 = x + 1;
            if (x2 < cols) {
                // not end of row yet
                pfgs = &fit_str[x2][y];
                elev = pfgs->elevation;
                flag = pfgs->flag;
                //while ( elev == M_NEG_ELEV ) {
                while ( !flag ) {
                    x2++;
                    if (x2 < cols) {
                        pfgs = &fit_str[x2][y];
                        elev = pfgs->elevation;
                        flag = pfgs->flag;
                    } else {
                        //check_me();
                        break;
                    }
                }
                if (x2 < cols) {    // got SECOND point of triange
                    lon = (ox + (x2 * col_step)) / 3600.0;
                    lat = (oy + (y * row_step)) / 3600.0;
                    alt = (double)elev;
                    t.p2 = SGVec3d(lon,lat,alt); // got SECOND point of triange
                    t.p2_x = x2;
                    t.p2_y = y;
                    SET_POINTV(1,x2,y);
                    // =========
                    // got two closest points in this row
                    y2 = y + 1; // go down a row, and look for third point
                    // x2 = x;     // start at current column - but maybe not
                    // ... x ... x ... - got these two points
                    // Maybe seek for third, should start at beginning 
                    x2 = 0;
                    fnd = 0;
                    if (y2 < rows) {
                        pfgs = &fit_str[x2][y2];
                        elev = pfgs->elevation;
                        flag = pfgs->flag;
                        while ( !flag ) {
                            x2++;
                            if (x2 < cols) {
                                pfgs = &fit_str[x2][y2];
                                elev = pfgs->elevation;
                                flag = pfgs->flag;
                            } else {
                                x2 = 0;
                                x2--;
                                y2++;
                                if ( y2 >= rows )
                                    break;
                            }
                        }
                        while (flag) { // got THIRD point of triange
                            lon = (ox + (x2 * col_step)) / 3600.0;
                            lat = (oy + (y2 * row_step)) / 3600.0;
                            alt = (double)elev;
                            t.p3 = SGVec3d(lon,lat,alt); // got THIRD point of triange
                            t.p3_x = x2;
                            t.p3_y = y2;
                            SET_POINTV(2,x2,y2);
                            add_tri(t);
                            x2++;
                            flag = 0;
                            if (!(x2 < cols)) {
                                x2 = 0;
                                y2++;
                            }
                            if ((x2 < cols)&&(y2 < rows)) {
                                pfgs = &fit_str[x2][y2];
                                elev = pfgs->elevation;
                                flag = pfgs->flag;
                                while ( !flag ) {
                                    x2++;
                                    if (x2 < cols) {
                                        pfgs = &fit_str[x2][y2];
                                        elev = pfgs->elevation;
                                        flag = pfgs->flag;
                                    } else {
                                        x2 = 0;
                                        y2++;
                                        if ( y2 >= rows )
                                            break;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        x++;    // go to NEXT column
        if ( x >= cols ) {
            x = 0;  // go to NEXT row
            y++;
            if ( y >= rows )
                break;
        }
    }
    // check all points are part of a triangle
    flag = 0;
    for (x = 0; x < cols; x++) {
        for (y = 0; y < rows; y++) {
            pfgs = &fit_str[x][y];
            elev = pfgs->elevation;
            flag = pfgs->flag;
            //if (elev != M_NEG_ELEV) {
            if (flag) {
                if (flag & (pvFLG_TRI1 | pvFLG_TRI2 | pvFLG_TRI3)) {
                    // valid point, is part of a triangle
#ifdef ADD_DEBUG_OUTPUT1
                    sprintf(buf,"x%04d:y%04d: Used %02d ",
                        x, y,
                        pfgs->used);
                    if (flag & pvFLG_TRI1)
                        strcat(buf,"TRI1 ");
                    if (flag & pvFLG_TRI2)
                        strcat(buf,"TRI2 ");
                    if (flag & pvFLG_TRI3)
                        strcat(buf,"TRI3 ");
                    sprtf("%s\n",buf);
#endif
                } else {
                    sprtf("x%04d:y%04d: Used %02d EEK: point x,y NOT marked!\n",
                        x, y, pfgs->used );
                }
            }
        }
    }
    max = fit_tris.size();
    sprtf( "Built %d tris from ...\n", max );
#ifdef ADD_DEBUG_OUTPUT2
    for (i = 0; i < max; i++) {
        t = tris[i];
        sprtf( "%d: %3.8f %2.8f %d - %3.8f %2.8f %d - %3.8f %2.8f %d\n",
            (i + 1),
            t.p1.x(),t.p1.y(),(int)t.p1.z(),
            t.p2.x(),t.p2.y(),(int)t.p3.z(),
            t.p3.x(),t.p3.y(),(int)t.p3.z() );
    }
#endif // #ifdef ADD_DEBUG_OUTPUT2
}

#if 0   // first attempt - NO GOOD
void PVArray::extract_tris(void)
{
    PpvFGS    pfgs;
    pvTRIV    t;
    int cols = get_cols();
    int rows = get_rows();
    double col_step = get_col_step();
    double row_step = get_row_step();
    double ox = get_originx();
    double oy = get_originy();
    double lon,lat,alt;
    int x, y, elev;
    int x2, y2, max;
    int i, fnd;
    unsigned int flag;
    //char * buf = _s_tmpbuf;
    x = 0;
    y = 0;
    while(1) {
        pfgs = &fit_str[x][y];
        elev = pfgs->elevation;
        flag = pfgs->flag;
        //while ( elev == M_NEG_ELEV ) {
        while ( flag == 0 ) {
            x++;
            if (x < cols) {
                pfgs = &fit_str[x][y];
                elev = pfgs->elevation;
                flag = pfgs->flag;
            } else {
                // check_me();
                break;  // out of points in this ROW
            }
        }
        if ( x < cols ) {
            lon = (ox + (x * col_step)) / 3600.0;
            lat = (oy + (y * row_step)) / 3600.0;
            alt = (double)elev;
            t.p1 = SGVec3d(lon,lat,alt); // got FIRST point of triange
            t.p1_x = x;
            t.p1_y = y;
            x2 = x + 1;
            if (x2 < cols) {
                // not end of row yet
                pfgs = &fit_str[x2][y];
                elev = pfgs->elevation;
                flag = pfgs->flag;
                //while ( elev == M_NEG_ELEV ) {
                while ( flag == 0 ) {
                    x2++;
                    if (x2 < cols) {
                        pfgs = &fit_str[x2][y];
                        elev = pfgs->elevation;
                        flag = pfgs->flag;
                    } else {
                        //check_me();
                        break;
                    }
                }
                if (x2 < cols) {
                    lon = (ox + (x2 * col_step)) / 3600.0;
                    lat = (oy + (y * row_step)) / 3600.0;
                    alt = (double)elev;
                    t.p2 = SGVec3d(lon,lat,alt); // got SECOND point of triange
                    t.p2_x = x2;
                    t.p2_y = y;
                    // =========
                    y2 = y + 1;
                    if (y2 < rows) {
                        pfgs = &fit_str[x][y2];
                        elev = pfgs->elevation;
                        flag = pfgs->flag;
                        //while ( elev == M_NEG_ELEV ) {
                        while ( flag == 0 ) {
                            y2++;
                            if (y2 < rows) {
                                pfgs = &fit_str[x][y2];
                                elev = pfgs->elevation;
                                flag = pfgs->flag;
                            } else {
                                //check_me();
                                break;
                            }
                        }
                        if (y2 < rows) {
                            lon = (ox + (x * col_step)) / 3600.0;
                            lat = (oy + (y2 * row_step)) / 3600.0;
                            alt = (double)elev;
                            t.p3 = SGVec3d(lon,lat,alt); // got THIRD point of triange
                            t.p3_x = x;
                            t.p3_y = y2;
                            // ===============
                            // set TRI flags
                            pfgs = &fit_str[t.p1_x][t.p1_y];
                            pfgs->flag |= pvFLG_TRI1;
                            pfgs->used++;

                            pfgs = &fit_str[t.p2_x][t.p2_y];
                            pfgs->flag |= pvFLG_TRI2;
                            pfgs->used++;

                            pfgs = &fit_str[t.p3_x][t.p3_y];
                            pfgs->flag |= pvFLG_TRI3;
                            pfgs->used++;
                            // ------------------
                            tris.push_back(t);  // store it
                            // ------------------
                        } else {
                            //check_me();
                            /// hmmm, got two points along the row
                            // MUST find a 3rd somewhere - try backward, and down
                            y2 = y + 1;
                            if (x > 0) {
                                x2 = x;
                                fnd = 0;
                                while( !fnd && (x2 > 0)) {
                                    x2--;       // back up 1 column
                                    pfgs = &fit_str[x2][y2];
                                    elev = pfgs->elevation;
                                    flag = pfgs->flag;
                                    if (flag)
                                        fnd = 1;
                                }
                                if (fnd) {
                                    lon = (ox + (x2 * col_step)) / 3600.0;
                                    lat = (oy + (y2 * row_step)) / 3600.0;
                                    alt = (double)elev;
                                    t.p3 = SGVec3d(lon,lat,alt); // got THIRD point of triange
                                    t.p3_x = x2;
                                    t.p3_y = y2;
                                    // ===============
                                    // set TRI flags
                                    pfgs = &fit_str[t.p1_x][t.p1_y];
                                    pfgs->flag |= pvFLG_TRI1;
                                    pfgs->used++;

                                    pfgs = &fit_str[t.p2_x][t.p2_y];
                                    pfgs->flag |= pvFLG_TRI2;
                                    pfgs->used++;

                                    pfgs = &fit_str[t.p3_x][t.p3_y];
                                    pfgs->flag |= pvFLG_TRI3;
                                    pfgs->used++;
                                    // ------------------
                                    tris.push_back(t);  // store it
                                    // ------------------
                                }
                            } else {
                                dbg_check();
                            }
                        }
                    }
                }
            }
        }
        x++;    // go to NEXT column
        if ( x >= cols ) {
            x = 0;  // go to NEXT row
            y++;
            if ( y >= rows )
                break;
        }
    }
    // check all points are part of a triangle
    flag = 0;
    for (x = 0; x < cols; x++) {
        for (y = 0; y < rows; y++) {
            pfgs = &fit_str[x][y];
            elev = pfgs->elevation;
            flag = pfgs->flag;
            //if (elev != M_NEG_ELEV) {
            if (flag) {
                if (flag & (pvFLG_TRI1 | pvFLG_TRI2 | pvFLG_TRI3)) {
                    // valid point, is part of a triangle
#ifdef ADD_DEBUG_OUTPUT1
                    sprintf(buf,"x%04d:y%04d: Used %02d ",
                        x, y,
                        pfgs->used);
                    if (flag & pvFLG_TRI1)
                        strcat(buf,"TRI1 ");
                    if (flag & pvFLG_TRI2)
                        strcat(buf,"TRI2 ");
                    if (flag & pvFLG_TRI3)
                        strcat(buf,"TRI3 ");
                    sprtf("%s\n",buf);
#endif
                } else {
                    sprtf("x%04d:y%04d: Used %02d EEK: point x,y NOT marked!\n",
                        x, y, pfgs->used );
                }
            }
        }
    }
    max = tris.size();
    sprtf( "Built %d tris from ...\n", max );
#ifdef ADD_DEBUG_OUTPUT2
    for (i = 0; i < max; i++) {
        t = tris[i];
        sprtf( "%d: %3.8f %2.8f %d - %3.8f %2.8f %d - %3.8f %2.8f %d\n",
            (i + 1),
            t.p1.x(),t.p1.y(),(int)t.p1.z(),
            t.p2.x(),t.p2.y(),(int)t.p3.z(),
            t.p3.x(),t.p3.y(),(int)t.p3.z() );
    }
#endif // #ifdef ADD_DEBUG_OUTPUT2
}
#endif // 0
void PVArray::fit_to_array_grid(void)
{
    int i, max;
    double lon, lat, alt, x, y;
    double xlocal, ylocal;
    int xindex, yindex, elev;
    PpvFGS  pfgs;
    int points_done = 0;

    max = fitted_list.size();
    for (i = 0; i < max; i++) {
        Point3D p = fitted_list[i];
        lon = p.lon();
        lat = p.lat();
        alt = p.elev();
        x = lon * 3600.0;
        y = lat * 3600.0;
        xlocal = (x - originx) / col_step;
        ylocal = (y - originy) / row_step;
        xindex = (int)(xlocal);
        yindex = (int)(ylocal);
        elev   = (int)alt;
        pfgs   = &fit_str[xindex][yindex];
        if (pfgs->flag & pvFLG_SET) {
            int ialt = pfgs->elevation;
            if (elev != ialt) {
                if (pvVERB3) sprtf("%d:%d: Previously set to %d, now %d\n",
                    xindex, yindex,
                    ialt, elev );
            }
            pfgs->flag |= pvFLG_SET2;
        } else {
            points_done++;
        }
        pfgs->elevation = elev;         // set the ELEVATION
        pfgs->flag     |= pvFLG_SET;    // add the SET flag
    }
    // check CORNERS exist???
    // TOP LEFT
    x = lon = originx;
    y = lat = originy;
    alt = altitude_from_grid( lon, lat );
    xlocal = (x - originx) / col_step;
    ylocal = (y - originy) / row_step;
    xindex = (int)(xlocal);
    yindex = (int)(ylocal);
    elev   = (int)alt;
    pfgs   = &fit_str[xindex][yindex];
    if ( !(pfgs->flag & pvFLG_SET) ) {
        points_done++;
        pfgs->elevation = elev;         // set the ELEVATION
        pfgs->flag     |= pvFLG_SET;    // add the SET flag
        sprtf( "Added TOP    LEFT  % 4d\n", elev );
    }

    // BOTTOM RIGHT
    x = lon = originx + ((cols - 1) * col_step);
    y = lat = originy + ((rows - 1) * row_step);
    alt = altitude_from_grid( lon, lat );
    xlocal = (x - originx) / col_step;
    ylocal = (y - originy) / row_step;
    xindex = (int)(xlocal);
    yindex = (int)(ylocal);
    elev   = (int)alt;
    pfgs   = &fit_str[xindex][yindex];
    if ( !(pfgs->flag & pvFLG_SET) ) {
        points_done++;
        pfgs->elevation = elev;         // set the ELEVATION
        pfgs->flag     |= pvFLG_SET;    // add the SET flag
        sprtf( "Added BOTTOM RIGHT % 4d\n", elev );
    }

    // BOTTOM LEFT
    x = lon = originx;
    y = lat = originy + ((rows - 1) * row_step);
    alt = altitude_from_grid( lon, lat );
    xlocal = (x - originx) / col_step;
    ylocal = (y - originy) / row_step;
    xindex = (int)(xlocal);
    yindex = (int)(ylocal);
    elev   = (int)alt;
    pfgs   = &fit_str[xindex][yindex];
    if ( !(pfgs->flag & pvFLG_SET) ) {
        points_done++;
        pfgs->elevation = elev;         // set the ELEVATION
        pfgs->flag     |= pvFLG_SET;    // add the SET flag
        sprtf( "Added BOTTOM LEFT  % 4d\n", elev );
    }

    // TOP RIGHT
    x = lon = originx + ((cols - 1) * col_step);
    y = lat = originy;
    alt = altitude_from_grid( lon, lat );
    xlocal = (x - originx) / col_step;
    ylocal = (y - originy) / row_step;
    xindex = (int)(xlocal);
    yindex = (int)(ylocal);
    elev   = (int)alt;
    pfgs   = &fit_str[xindex][yindex];
    if ( !(pfgs->flag & pvFLG_SET) ) {
        points_done++;
        pfgs->elevation = elev;         // set the ELEVATION
        pfgs->flag     |= pvFLG_SET;    // add the SET flag
        sprtf( "Added TOP    RIGHT % 4d\n", elev );
    }

    sprtf("Set %d points, of %d in fitted list.\n", points_done, max );
    extract_tris();
}


PVArray::~PVArray( void ) {
    sprtf("class PVArray destructor called.\n");
    delete [] in_data;
    // delete [] out_data;
    free_mem();
    PVArray::close();
}

// eof - poly-array.cxx

