
// win_bmp.c - part of gshhs (for WIN32) project
//
// PURPOSE: Collect lat,lon points into the iBlock array,
// and generate a BITMAP file, showing those points.
// 2008/02/05 - geoff mclane 
//
#include "gshhs_win.h"

#define  DEF_SCALE   8
#define  VERT_SCALE  1
#define  DEF_HEIGHT  181*DEF_SCALE*VERT_SCALE
#define  DEF_WIDTH   361*DEF_SCALE

BYTE iBlock[DEF_HEIGHT][DEF_WIDTH] = {0};
long min_ilon, max_ilon, min_ilat, max_ilat;
//BOOL g_bUseMercator = FALSE;   // use a FLAT world, where SIZE is exagerated
// else use non mercator projection
// y = dlat
// x = dlon*cos(dlat)

void Clear_lon_lat( PGSINFO pgsi )
{
   if(pgsi) {
      PBMPINFO pbmi = &pgsi->bmpinfo;
      PGSRANGE pgsr = &pbmi->range;
      pgsr->maxilat = -1;
      pgsr->minilat = 2147483647;
      pgsr->maxilon = -1;
      pgsr->minilon = 2147483647;
      pgsr->mxlat = -90.0;
      pgsr->mnlat = 90.0;
      pgsr->mxlon = -180.0;
      pgsr->mnlon = 180.0;
   }

   ZeroMemory(&iBlock, sizeof(iBlock));
   min_ilon = DEF_WIDTH + 16;
   max_ilon = -min_ilon;
   min_ilat = DEF_HEIGHT + 16;
   max_ilat = -min_ilat;
}

void Get_iLat_iLon( PGSINFO pgsi, double lat, double lon, int * pilat, int * pilon )
{
   int ilon = (int)(((lon + 180.0) * (double)DEF_SCALE) + 0.5); // range 0 to 360
   int ilat = (int)(((lat +  90.0) * (double)DEF_SCALE * (double)VERT_SCALE) + 0.5); // range 0 to 180
   if( !g_bUseMercator ) {
      double y = lat;
      double x = lon * cos( SGD_DEGREES_TO_RADIANS * lat );
      ilon = (int)(((x + 180.0) * (double)DEF_SCALE) + 0.5); // range 0 to 360
   }

   *pilat = ilat;
   *pilon = ilon;
}

//   int ilon = (int)(((lon + 180.0) * (double)DEF_SCALE) + 0.5); // range 0 to 360
//   int ilat = (int)(((lat +  90.0) * (double)DEF_SCALE) + 0.5); // range 0 to 180
//   if( !g_bUseMercator ) {
//      double y = lat;
//      double x = lon * cos( SGD_DEGREES_TO_RADIANS * lat );
//      ilon = (int)(((x + 180.0) * (double)DEF_SCALE) + 0.5); // range 0 to 360
//   }


void Add_Lat_Lon_Point( PGSINFO pgsi, double lat, double lon, BYTE b, BOOL LimChk,
                       BOOL replace )
{
   int ilat, ilon;
   Get_iLat_iLon( pgsi, lat, lon, &ilat, &ilon );
   if(( ilat >= 0 ) && (ilat < DEF_HEIGHT) &&
      ( ilon >= 0 ) && (ilon < DEF_WIDTH))
   {
      PBYTE pb = &iBlock[ilat][ilon];
      if(LimChk) {
         if(( ilat >= min_ilat ) && (ilat <= max_ilat) &&
            ( ilon >= min_ilon ) && (ilon <= max_ilon))
         {
            if(( *pb == 0 ) || replace )
               *pb = b;
         }
      } else {
         // add em all
         if(( *pb == 0 ) || replace )
            *pb = b;
      }
   } else {
      printf( "WARNING: lat %10.5f, lon %10.5f OUTSIDE (%d,%d)\n",
         lat, lon, ilat, ilon );
   }
}

BOOL In_Special( PGSINFO pgsi, double lat, double lon )
{
   double maxlat = 60.0;    //51.0;
   double minlat = 40.0;    //
   double minlon =  0.0;    // 7
   double maxlon = 20.0;    // 17.0;
   if(( lat >= minlat )&&
      ( lat <= maxlat )&&
      ( lon >= minlon )&&
      ( lon <= maxlon ))
      return TRUE;

   return FALSE;
}

void Add_lon_lat( double lastlon, double lastlat, PGSINFO pgsi )
{
   // lon x = -180 to 180 = 360
   // lat y =   90 to -90 = 180
   int ilat, ilon;
   Get_iLat_iLon( pgsi, lastlat, lastlon, &ilat, &ilon );
   if(( ilat >= 0 ) && (ilat < DEF_HEIGHT) &&
      ( ilon >= 0 ) && (ilon < DEF_WIDTH))
   {
      if( ilat < min_ilat )
         min_ilat = ilat;
      if( ilat > max_ilat )
         max_ilat = ilat;
      if( ilon < min_ilon )
         min_ilon = ilon;
      if( ilon > max_ilon )
         max_ilon = ilon;

      iBlock[ilat][ilon] = 1;

   } else {
      printf( "WARNING: lat %10.5f, lon %10.5f OUTSIDE (%d,%d)\n",
         lastlat, lastlon, ilat, ilon );
   }
}

BOOL iLat_in_Range( int ilat )
{
   if(( ilat >= 0 ) && (ilat < DEF_HEIGHT))
      return TRUE;

   return FALSE;
}
BOOL iLon_in_Range( int ilon )
{
   if(( ilon >= 0 ) && (ilon < DEF_WIDTH))
      return TRUE;

   return FALSE;
}

BOOL  Limits_in_Range( PGSINFO pgsi )
{
   if(iLat_in_Range( max_ilat )&&
      iLat_in_Range( min_ilat )&&
      iLon_in_Range( max_ilon )&&
      iLon_in_Range( min_ilon )&&
      ( min_ilat < max_ilat )&&
      ( min_ilon < max_ilon ))
      return TRUE;

   return FALSE;
}

void Put_Number( PBYTE pn, int ilat, int ilon )
{
   int   x, y, off;
   PBYTE pb;
   BYTE val;
   // 9 bits high
   for( y = 8; y >= 0; y-- )
   {
      pb = &iBlock[ilat][ilon];
      // 7 bits wide
      for( x = 0; x < 7; x++ )
      {
         off = (y * 7) + x;
         val = pn[off];
         if(val) {
            pb[x] = 3;
         }
      }
      ilat++;  // to next bitmap row
   }
}

// 7 x 9 bits
void Add_Lat_Numbers( PGSINFO pgsi, double lon )
{
   PBYTE pn;
   double lat;
   int ilat, ilon, inum;
   //for( lat = -80.0; lat <= 80.0; lat += 10.0 )
   for( lat = -90.0; lat <= 90.0; lat += 10.0 )
   {
      Get_iLat_iLon( pgsi, lat, lon, &ilat, &ilon );
      ilat -= 4;  // straddle the number over the lat line
      ilon += 4;  // shift it right, off the lon line
      if( lat < 0.0 ) {
         pn = Get_Num_Array( 10 );  // get the MINUS sign
         if(pn) {
            Put_Number( pn, ilat, ilon );
            ilon += 8;
         }
         inum = (int)(-lat / 10);
      } else {
         inum = (int)(lat / 10);
      }
      pn = Get_Num_Array( inum );
      if(pn) {
         Put_Number( pn, ilat, ilon );
         ilon += 8;
      }
      pn = Get_Num_Array( 0 );
      if(pn) {
         Put_Number( pn, ilat, ilon );
         ilon += 8;
      }
   }
}

void Add_Lon_Numbers( PGSINFO pgsi, double lat )
{
   PBYTE pn;
   double lon;
   int ilat, ilon, inum1, inum2;
   //for( lon = -170.0; lon <= 170.0; lon += 10.0 )
   for( lon = -180.0; lon <= 180.0; lon += 10.0 )
   {
      Get_iLat_iLon( pgsi, lat, lon, &ilat, &ilon );
      ilat += 6;  // move above lat line
      ilon -= 10; // straddle the lon line
      if( lon < 0.0 ) {
         pn = Get_Num_Array( 10 );
         if(pn) {
            ilon -= 8;  // back up more for negative numbers
            Put_Number( pn, ilat, ilon );
            ilon += 8;
         }
         inum1 = (int)(-lon / 10);
      } else {
         inum1 = (int)(lon / 10);
      }
      inum2 = inum1 / 10;  // from 17 or 7 = 1 or 0
      inum1 = inum1 % 10;
      if( inum2 ) {
         pn = Get_Num_Array( inum2 );
         if(pn) {
            Put_Number( pn, ilat, ilon );
            ilon += 8;
         }
      }
      pn = Get_Num_Array( inum1 );
      if(pn) {
         Put_Number( pn, ilat, ilon );
         ilon += 8;
      }
      pn = Get_Num_Array( 0 );
      if(pn) {
         Put_Number( pn, ilat, ilon );
         ilon += 8;
      }
   }
}


void Add_Lat_Lon_Numbers( PGSINFO pgsi )
{
   Add_Lat_Numbers( pgsi, -170.0 );
   Add_Lat_Numbers( pgsi, -90.0 );
   Add_Lat_Numbers( pgsi, 0.0 );
   Add_Lat_Numbers( pgsi, 90.0 );
   Add_Lat_Numbers( pgsi, 170.0 );
   Add_Lon_Numbers( pgsi, -40.0 );
   Add_Lon_Numbers( pgsi, 0.0 );
   Add_Lon_Numbers( pgsi, 40.0 );
   if( g_bUseMercator ) {
      // only ADD these far north and far south
      // if using MERCATOR distortion
      Add_Lon_Numbers( pgsi, -80.0 );
      Add_Lon_Numbers( pgsi, 80.0 );
   }
}

void Put_Bit_Array( PBYTE pn, int ilat, int ilon, int mx, int my )
{
   int   x, y, off;
   PBYTE pb;
   BYTE val;
   // 11 bits high
   for( y = my - 1; y >= 0; y-- )
   {
      pb = &iBlock[ilat][ilon];
      // 7 bits wide
      for( x = 0; x < mx; x++ )
      {
         off = (y * mx) + x;
         val = pn[off];
         if(val) {
            pb[x] = 3;
         }
      }
      ilat++;  // to next bitmap row
   }
}

void  Add_Text_to_World( PGSINFO pgsi, int ilat, int ilon, char * txt, int len )
{
   int i;
   int mx, my;
   // ASCII chars are 7 x 11
   Get_Char_Bit_Size( &mx, &my ); // 7 x 11, or whatever
   for( i = 0; i < len; i++ )
   {
      PBYTE pb = Get_Char_Bit_Array( txt[i] );
      Put_Bit_Array( pb, ilat, ilon, mx, my );
      ilon += 8;
   }
}

static char szTmpBuff[1024];
void Add_Notice_to_World( PGSINFO pgsi )
{
   size_t len;
   if( Limits_in_Range( pgsi ) ) {
      char * pn = szTmpBuff;
      time_t ltime;
      char timebuf[26];
      POINT pt;
      // Get UNIX-style time and display as string. 
      time( &ltime );
      ctime_s(timebuf, 26, &ltime);
      len = 26;
      while(len--) {
         if(timebuf[len] > ' ')
            break;
         timebuf[len] = 0;
      }
      sprintf(pn, "Geoff McLane - Generated %s, by gshhs_win.exe",
         timebuf);
      len = strlen(pn);
      pt.y = max_ilat - 18;
      pt.x = min_ilon + 10;
      Add_Text_to_World( pgsi, pt.y, pt.x, pn, (int)len );
      sprtf( "TEXT[%s] at %d,%d"MEOR, pn, pt.x, pt.y );

      sprintf(pn, "From file [%s], with %ld headers, ",
         pgsi->mv.fn, pgsi->hdrcnt );
      sprintf(EndBuf(pn), "and %ld points.", pgsi->ptcnt );
      len = strlen(pn);
      pt.y -= 11;
      Add_Text_to_World( pgsi, pt.y, pt.x, pn, (int)len );
      sprtf( "TEXT[%s] at %d,%d"MEOR, pn, pt.x, pt.y );

   }
}

// add lat. lon lines to bitmap
// ============================
void  Add_Lat_Lon_Lines( PGSINFO pgsi )
{
   double lat, lon;
   double dlat, dlon;
   double x, y;
   BOOL  chklim = Limits_in_Range( pgsi );
   for( lat = -90.0; lat < 90.0; lat += 10.0 )
   {
      for( lon = -180.0; lon <= 180.0; lon += 10.0 )
      {
         dlat = lat + 10;
         dlon = lon + 10;
         // fixed lon, do lat - vertical lines
         for( y = lat; y <= dlat; y += 0.25 )
            Add_Lat_Lon_Point( pgsi, y, lon, 2, chklim, FALSE );

         if( lon < 180.0 ) {
            // fixed lat, do lon - horizontal lines
            for( x = lon; x <= dlon; x += 0.25 )
               Add_Lat_Lon_Point( pgsi, lat, x, 2, chklim, FALSE );
         }
      }
   }
   Add_Lat_Lon_Numbers( pgsi );
}

void Add_Min_Max_Border( PGSINFO pgsi )
{
   int x, y;
   if( Limits_in_Range( pgsi ) ) {
      sprtf( "Adding border, lefttop(%d,%d), rightbottom(%d,%d) ...\n",
         max_ilat, min_ilon,
         min_ilat, max_ilon );
      // from top to bottom lat, set max, min lon
      for( y = min_ilat; y <= max_ilat; y++ )
      {
         // vertical lines
         iBlock[y][min_ilon] = 1;
         iBlock[y][max_ilon] = 1;
      }

      // from left to right lon, set max, min lat
      for( x = min_ilon; x <= max_ilon; x++ )
      {
         // horizontal lines
         iBlock[min_ilat][x] = 1;
         iBlock[max_ilat][x] = 1;
      }
   } else {
      sprtf( "WARNING: No border due, %d,%d %d,%d out of range...\n",
         max_ilat, min_ilon,
         min_ilat, max_ilon );
   }
   Add_Notice_to_World( pgsi );
}

static BITMAPFILEHEADER _s_hdr;
static BITMAPINFOHEADER _s_bi;

int NormalizeBPP( int nBPP )
{
   int iBPP = 24;
	if( nBPP <= 1 )
		iBPP = 1;
	else if( nBPP <= 4 )
		iBPP = 4;
	else if( nBPP <= 8 )
		iBPP = 8;
	else
		iBPP = 24;
   return iBPP;
}

// Only certain bitmaps have a color count
// That is a COLORREF table
int	GetColorCnt( int nBPP )
{
	int ClrCnt = 0;   // others have NONE
   int iBPP = NormalizeBPP(nBPP);
	switch( iBPP ) {
	case 1: ClrCnt = 2; break;
	case 4: ClrCnt = 16; break;
	case 8: ClrCnt = 256; break;
	}
	return ClrCnt;
}

size_t GetPaletteSize( int nBPP )
{
   return (GetColorCnt(nBPP) * sizeof(RGBQUAD));
}

DWORD GetRowWidth( DWORD dw, DWORD nBPP )
{
	// OK, how we calculate the DATA size
	// depends on the Bit count
	DWORD adw = dw;	// Set DEFAULT to 8 - Not correct, but what to do???
   DWORD iBPP = NormalizeBPP(nBPP);
	switch( iBPP )
	{
	case 1:		// Monochrome
		adw = dw / 8;	// Each bit in each byte is a colour ON/OFF
		if( dw % 8 )
			adw++;
		break;
	case 4:		// 16 Colours
		adw = dw / 2;	// Each nibble (4-bits) is a color index
		if( dw % 2 )
			adw++;
		break;
	case 8:		// 256 Colours
		adw = dw;		// Each BYTE (8-bits) is a color index
		break;
	case 24:	// 24 Bit Colour
		adw = dw * 3;	// Each THREE (3) bytes is a COLOR
		break;
	}

	// Now onto 32-bit BOUNDARY
	if( adw % 4 )
		adw = ((adw / 4) + 1) * 4;

   return adw; // adjusted ROW width, in BYTES
}


void InitBitmapInfoHeader( LPBITMAPINFOHEADER pih, DWORD dwWid, DWORD dwHt, int nBPP )
{
	int		iBPP;
	//dv_fmemset( lpBmInfoHdr, 0, sizeof (BITMAPINFOHEADER) );
	ZeroMemory( pih, sizeof (BITMAPINFOHEADER) );

	pih->biSize        = sizeof(BITMAPINFOHEADER);
	pih->biWidth       = dwWid;
	pih->biHeight      = dwHt;
	pih->biPlanes      = 1;

	// Fix Bits per Pixel
   iBPP = NormalizeBPP(nBPP);

	pih->biBitCount    = iBPP;
	pih->biSizeImage   = (WIDTHBYTES(dwWid * iBPP) * dwHt); // total BYTE image block
   pih->biCompression = BI_RGB;  // assume RGB output
}

void SetMonochromeMap( RGBQUAD * prq )
{
   Set_BMP_1BPP( prq ); // in gshhs_color.c
}

void SetBPP4Map( RGBQUAD * prq )
{
   Set_BMP_4BPP( prq ); // in gshhs_color.c
}

void SetBPP8Map( RGBQUAD * prq )
{
   // 256 byte color map
   Set_BMP_8BPP(prq);  // in gshhs_color.c
}


int Write_lon_lat( char * pfile, PGSINFO pgsi )
{
   int iret = 0;  // assume FAILED
   int width  = DEF_WIDTH;
   int height = DEF_HEIGHT;
   int nBPP   = pgsi->bmpinfo.nBPP; // = BMP_DEF_BPP, or as requested;
   FILE * fp = fopen( pfile, "wb" );
   if(fp) {
      int x, y;
      size_t palsz = GetPaletteSize(nBPP);
      BITMAPFILEHEADER * phdr = &_s_hdr;
      LPBITMAPINFOHEADER pbi  = &_s_bi;
      int cols = GetRowWidth( width, nBPP );   // per BPP and rounded to 4 bytes
      int rows = height;
      long bufsize = cols * rows;   // to HOLD the BITS
      size_t dibsz = sizeof(BITMAPINFOHEADER) + palsz + bufsize;
      size_t size  = sizeof(BITMAPFILEHEADER) + dibsz;
      PBYTE pbits;
      PBYTE pout;
      RGBQUAD * ppal = 0; // if ANY
      int row, col;
      BYTE  b;

      pbits = (PBYTE)MALLOC( bufsize + palsz );
      if( !pbits ) {
         sprtf( "ERROR: Memory failed on %u bytes ...\n", bufsize );
         fclose(fp);
         goto FAILED;
      }

      if( palsz ) {
         ppal = (RGBQUAD *)( pbits + bufsize );
         // make a PALETTE of COLOURS
         switch( nBPP )
         {
         case 1:  // monochrome
            SetMonochromeMap(ppal);
            break;
         case 4:
            SetBPP4Map(ppal);
            break;
         default:
            fclose(fp);
            free(pbits);
            sprtf( "WARNING: BPP size of %d NOT HANDLED\n", nBPP );
            goto FAILED;
         }
      }

      if( g_AddBorder )
         Add_Min_Max_Border( pgsi );
      if( g_AddGrid )
         Add_Lat_Lon_Lines( pgsi );

      // BITMAP SETUP
      // Bits
      for( y = 0; y < DEF_HEIGHT; y++ )
      {
         row = y;
         pout = &pbits[(row * cols)];  // get ROW pointer
         if( nBPP == 4 ) {
            int nib = 0;
            int ind = 0;
            BYTE cv;
            for( x = 0; x < DEF_WIDTH; x++ )
            {
               b = iBlock[y][x]; // get color
               col = x / 2;   // get COLUMN offset (4-bits)
               cv = pout[col];   // extract the 2 nibble index
               ind = WHITE_IND;  // set WHITE
               if( b ) {
                  if( b == 3 ) {
                     ind = RED_IND; // set RED
                  } else if( b == 2 ) {
                     ind = GRAY_IND;   // set GRAY
                  } else {
                     ind = BLACK_IND;  // set BLACK
                  }
               }
               if(nib)
                  cv = (cv & 0xf0) | ind;
               else
                  cv = (ind << 4) | (cv & 0x0f);
               pout[col] = cv;   // put 2 nibble index
               // flip the NIB being set
               if(nib)
                  nib = 0;
               else
                  nib = 1;
            }
         } else {
            for( x = 0; x < DEF_WIDTH; x++ )
            {
               col = x * 3;   // get COLUMN offset (24-bits)
               b = iBlock[y][x];
               if( b ) {
                  if( b == 2 ) {
                     // set GRAY
                     pout[col+2] = 190;
                     pout[col+1] = 190;
                     pout[col+0] = 190;
                  } else {
                     // set BLACK
                     pout[col+2] = 0;
                     pout[col+1] = 0;
                     pout[col+0] = 0;
                  }
               } else {
                  // set WHITE
                  pout[col+2] = 0xff;
                  pout[col+1] = 0xff;
                  pout[col+0] = 0xff;
               }
            }
         }
      }
      // Info-header
      InitBitmapInfoHeader( pbi,    // LPBITMAPINFOHEADER lpBmInfoHdr,
                           width,
                           height,
                           nBPP ); // 	  int nBPP )
      // File Header
      phdr->bfType      = DIB_HEADER_MARKER;   // simple "BM" signature
      phdr->bfSize      = (DWORD)size;   // file size
      phdr->bfReserved1 = 0;
      phdr->bfReserved2 = 0;
      phdr->bfOffBits   = (DWORD)sizeof(BITMAPFILEHEADER) + pbi->biSize + palsz;

      // WRITE BITMAP
      fwrite(phdr, 1, sizeof(BITMAPFILEHEADER), fp);
      fwrite(pbi, 1, sizeof(BITMAPINFOHEADER), fp);
      if( ppal && palsz )
         fwrite( ppal, 1, palsz, fp );
      fwrite(pbits, 1, bufsize, fp);

      MFREE(pbits);   // free BIT BUFFER, and PALETTE buffer, if any
      fclose(fp);
      sprtf( "Written [%s] BMP file ...\n", pfile );
      iret = 1;
   } else {
FAILED:
      sprtf( "WARNING: Failed to write %s file ...\n", pfile );
   }

   return iret;
}

// extract the bit informaiont
//            Num Row Col
BYTE bitblocks[256][14][7];

void Bitmap_2_Bits(void)
{
   char * pfile = "ascii7x9.bmp"; // "tempacii.bmp";   // "tempnums.bmp";
   BITMAPFILEHEADER * phdr = &_s_hdr;
   LPBITMAPINFOHEADER pbi  = &_s_bi;
   size_t size = sizeof(BITMAPFILEHEADER);
   size_t size1;
   size_t size2 = sizeof(BITMAPINFOHEADER);
   size_t rdsz;
   PBYTE pb = NULL;
   size_t off;
   int width, height, rowwid, x, y, maxnum;
   PBYTE pbits;
   BYTE  r,g,b, val;
   FILE * fp = fopen( pfile, "rb" );
   if(fp) {
      size1 = fread( phdr, 1, size, fp );
      if( size1 == size ) {
         if( phdr->bfType == DIB_HEADER_MARKER ) {
            rdsz = fread( pbi, 1, size2, fp );
            if( rdsz == size2 ) {
               if( pbi->biBitCount == 24 ) {
                  if( pbi->biSizeImage ) {
                     width = pbi->biWidth;
                     height = pbi->biHeight;
                     rowwid = GetRowWidth( width, pbi->biBitCount );
                     sprtf( "Width %d, RowWidth %d bytes, Height %d\n",
                        width, rowwid, height );
                     sprtf( "Image size is %d bytes, calculated %d ..\n",
                        pbi->biSizeImage,
                        rowwid * height );
                     pb = (PBYTE)MALLOC(pbi->biSizeImage);
                     CHKMEM(pb,pbi->biSizeImage);
                     off = fseek( fp, phdr->bfOffBits, SEEK_SET );
                     rdsz = fread( pb, 1, pbi->biSizeImage, fp );
                     if( rdsz == pbi->biSizeImage ) {
                        int num, bit;
                        int maxy, miny, bitcnt;
                        r = 255;
//         for( icol = 0; icol < lpbih->biWidth; icol++ ) {  // proceed in COLUMNS
//            for( irow = lpbih->biHeight - 1; irow >= 0; irow-- ) { // process HEIGHT
//               PRGB24 p24 = (PRGB24) &lps[ (irow * imgrow) + (icol * sizeof(RGB24)) ];
//                        for( y = 0; y < height; y++ )
                        for( y = height - 1; y >= 0; y-- )
                        {
                           num = 0;
                           //sprtf( "Row %d of %d ...\n", (y + 1), height );
                           bit = 0;
                           for( x = 0; x < width; x++ )
                           {
                              num = x / 7;   // set the ASCII numbers - each 7 bits wide
                              pbits = &pb[(( y * rowwid ) + (x * 3))];
                              r = pbits[0];
                              g = pbits[1];
                              b = pbits[2];
                              if( r == 255 ) {
                                 val = 0;
                              } else {
                                 val = 1;
                              }
                              //        Num Row Col
                              bitblocks[num][y][bit] = val;
                              //sprtf( "Num %d, Row %d, Col %d, has value %d (x = %d)\n",
                              //   num, y+1, bit+1, val, x );
                              bit++;
                              if( bit == 7 )
                                 bit = 0;
                           }
                        }
                        maxnum = width / 7;
                        // get max and min - leave 1 free row top and bottom
                        maxy = -1;
                        miny = 1000;
                        for( num = 0; num < maxnum; num++ )
                        {
                           for( y = 14-1; y >= 0; y-- )
                           {
                              bitcnt = 0;
                              for( x = 0; x < 7; x++ )
                              {
                                 if( bitblocks[num][y][x] ) {
                                    bitcnt++;
                                 }
                              }
                              if( bitcnt ) {
                                 // bit on this row
                                 if( y > maxy )
                                    maxy = y;
                                 if( y < miny )
                                    miny = y;
                              }
                           }
                        }
                        sprtf( "Got maxy %d, and miny %d ...\n", maxy, miny );
                        if( maxy < 13 )
                           maxy++;
                        if( miny > 0 )
                           miny--;
                        sprtf( "Adjusted to maxy %d, and miny %d ...\n", maxy, miny );
                        sprtf( "// Arrays are 7 x %d ...\n", maxy + 1);

                        for( num = 0; num < maxnum; num++ )
                        {
                           val = num + ' ';
                           sprtf( "// #%d %d, %#x, (%c)\n", num, val, val, val );
                           sprtf( "static BYTE bits_ascii%d[] = {\n", num );
                           //for( y = 14-1; y >= 0; y-- )
                           //for( y = 11; y >= 3; y-- )
                           //for( y = 10; y >= 2; y-- )
                           for( y = maxy; y >= miny; y-- )
                           {
                              for( x = 0; x < 7; x++ )
                              {
                                 if( x == 0 )
                                    sprtf( " " );
                                 if( bitblocks[num][y][x] ) {
                                    sprtf( "1" );
                                 } else {
                                    sprtf( "0" );
                                 }

                                 if( (y == miny) && (( x + 1) == 7) )
                                    sprtf( "  // end #%d %d, %#x (%c)", num, val, val, val );
                                 else
                                    sprtf( "," );

                                 if( (x + 1) == 7 )
                                    sprtf( "\n" );
                              }
                           }
                           sprtf( "};\n" );
                        }
                        sprtf( "// Array of ASCII blocks\n" );
                        sprtf( "static PBYTE ascci_array[] = {\n" );
                        for( num = 0; num < maxnum; num++ )
                        {
                           val = num + ' ';
                           sprtf( " bits_ascii%d", num );
                           if( (num + 1) < maxnum )
                              sprtf( ",  // %d %#x (%c)\n", val, val, val );
                           else
                              sprtf( "  // %d %#x (%c)\n", val, val, val );
                        }
                        sprtf( "};\n" );
                     } else {
                        sprtf( "ERROR: Read request %d, got %d ...\n",
                           pbi->biSizeImage, rdsz );
                     }
                     MFREE(pb);
                  }
               }
            } else {
               sprtf( "ERROR: Read request %d, got %d ...\n",
                  size2, rdsz );
            }
         }
      }
      fclose(fp);
   }
   pgm_exit(-1);
}


// eof - win_bmp.c
