// gshhs_bmp.c
// write a user BITMAP file ...

#include "gshhs_win.h"

// forward references
BOOL Pt_In_BMP_Boundary( PGSINFO pgsi, int ilat, int ilon );
BOOL  Pt_In_BMP_Range( PGSINFO pgsi, int ilat, int ilon );

static BITMAPFILEHEADER _s_hdr;
static BITMAPINFOHEADER _s_bi;
static BOOL gbAdjustText = TRUE;
static BOOL bAddLimitsBorder = TRUE;
static BOOL bAddMinMaxBorder = TRUE;
static BOOL bMinMaxOverwrite = FALSE;

// ================================================================
static BOOL gbReverseNib = FALSE;   // THIS ***SHOULD*** BE FALSE
// ================================================================
// diagnostic only
static BOOL bCheckTLBRVals = TRUE;
static BOOL bShowAnomolies = TRUE;
static BOOL bWriteBlockBMP = FALSE;
static BOOL g_bCullBitmap = TRUE;

static char szTmpBuff[1024];

// User's requested BITMAP
void BMP_iLatLon_2_latlon( PGSINFO pgsi, double * plat, double * plon, int inlat, int inlon )
{
   PBMPINFO pbmi = &pgsi->bmpinfo;
   double scale = (double)pbmi->range.scale;
   int ilat = inlat + pbmi->range.ailat; // adjust top offset
   int ilon = inlon + pbmi->range.ailon; // adjust left offset
   double lon = ((double)ilon / scale) - 180.0;
   double lat = ((double)ilat / scale) -  90.0;
   if( !pgsi->bmpinfo.mercator ) {
      lon = lon / cos( SGD_DEGREES_TO_RADIANS * lat );
   }
   *plat = lat;
   *plon = lon;
}

void Get_BMP_iLatLon( PGSINFO pgsi, double lat, double lon, int * pilat, int * pilon )
{
   PBMPINFO pbmi = &pgsi->bmpinfo;
   double scale = (double)pbmi->range.scale;
   //double alat = lat - pgsi->range.s;
   //double alon = lon - pgsi->range.w;
   int ilon = (int)(((lon + 180.0) * scale) + 0.5); // range 0 to 360
   //int ilat = (int)(((lat +  90.0) * scale * 2.0) + 0.5); // range 0 to 180
   int ilat = (int)(((lat +  90.0) * scale) + 0.5); // range 0 to 180
   if( !pgsi->bmpinfo.mercator ) {
      double y = lat;
      double x = lon * cos( SGD_DEGREES_TO_RADIANS * lat );
      ilon = (int)(((x + 180.0) * scale) + 0.5); // range 0 to 360
   }
   if( VERB9(pgsi) ) {
      sprtf( "v9: Lat %f, lon %f, translated to %d,%d = %d,%d ... (adj %d,%d)\n",
         lat, lon, ilat, ilon,
         ilat - pbmi->range.ailat, ilon - pbmi->range.ailon,
         pbmi->range.ailat, pbmi->range.ailon );
   }
   ilat -= pbmi->range.ailat; // adjust top offset
   ilon -= pbmi->range.ailon; // adjust left offset

   *pilat = ilat;
   *pilon = ilon;
}

void Add_2_User_BMP( PGSINFO pgsi, double lat, double lon, int id )
{
   if(pgsi) {
      PBMPINFO pbmi = &pgsi->bmpinfo;
      PGSRANGE pgsr = &pbmi->range;
      //if( In_Special( pgsi, lat, lon ) )
      //if(( lat >= pgsr->s )&&( lat <= pgsr->n )&&( lon >= pgsr->w )&&( lon <= pgsr->e ))
      if(( lat >= pgsr->s )&&( lat <= pgsr->n )&&( lon >= pgsr->w )&&( lon <= pgsr->e ))
      {
         int ilat, ilon;
         Get_BMP_iLatLon( pgsi, lat, lon, &ilat, &ilon );
         if( Pt_In_BMP_Range( pgsi, ilat, ilon ) ) {
         //if(( ilat >= 0 )&&( ilon >= 0)&&
         //   ( ilat < pbmi->range.height)&&
         //   ( ilon < pbmi->range.width))
         //if( bNoBoundaryCheck || Pt_In_BMP_Boundary( pgsi, ilat, ilon ) )
         //{
            // if FITS in the block BOUNDARIES
            PBYTE pb = pgsr->pblock;
            int   width = pbmi->range.width;
            if(pb) {
               long off = (ilat * width) + ilon;
               BYTE cb = pb[off];   // get CURRENT color byte

               if( pgsi->overwrite )
                  pb[off] = pgsi->def_bmc;   // set DEFAULT color (BYTE)
               else if( cb == BMC_WHITE )
                  pb[off] = pgsi->def_bmc;   // set DEFAULT color (BYTE)

               pgsr->ptcount++;  // bump the POINT count
               // fill in the LIMIT RANGES
               if( ilat > pgsr->maxilat )
                  pgsr->maxilat = ilat;
               if( ilat < pgsr->minilat )
                  pgsr->minilat = ilat;
               if( ilon > pgsr->maxilon )
                  pgsr->maxilon = ilon;
               if( ilon < pgsr->minilon )
                  pgsr->minilon = ilon;
               if( lat > pgsr->mxlat )
                  pgsr->mxlat = lat;
               if( lat < pgsr->mnlat )
                  pgsr->mnlat = lat;
               if( lon > pgsr->mxlon )
                  pgsr->mxlon = lon;
               if( lon < pgsr->mnlon )
                  pgsr->mnlon = lon;
               // ========================
               //if( bNoBoundaryCheck ) {
                  // then ADJUST boundaries
                  if( ilon < pgsr->TLilon )
                     pgsr->TLilon = ilon;
                  if( ilon > pgsr->BRilon )
                     pgsr->BRilon = ilon;
               //}
            }
         } else if (bShowAnomolies) {
            sprtf("WARNING: Lat %3.7f, Lon %3.7f, offsets y %d, x %d on %d,%d."MEOR,
               lat, lon, ilat, ilon, pbmi->range.width, pbmi->range.height );
            sprtf("DOES NOT FIT in TL %d,%d, BR %d,%d! WHY???"MEOR,
               pgsr->TLilat, pgsr->TLilon,
               pgsr->BRilat, pgsr->BRilon );
            if( ilat > pgsr->TLilat )
               sprtf( "ilat(%d) > TLilat(%d) ", ilat, pgsr->TLilat );
            if( ilat < pgsr->BRilat )
               sprtf( "ilat(%d) < BRilat(%d) ", ilat, pgsr->BRilat );
            if( ilon > pgsr->BRilon )
               sprtf( "ilon(%d) > BRilon(%d) ", ilon, pgsr->BRilon );
            if( ilon < pgsr->TLilon )
               sprtf( "ilon(%d) < TLilon(%d) ", ilon, pgsr->TLilon );
            if( ilat < 0 )
               sprtf( "ilat(%d) < ZERO ", ilat );
            if( ilon < 0 )
               sprtf( "ilon(%d) < ZERO ", ilon );
            if(ilat >= pgsr->height )
               sprtf( "ilat(%d) >= height(%d) ", ilat, pgsr->height );
            if(ilon >= pgsr->width )
               sprtf( "ilon(%d) >= width(%d) ", ilat, pgsr->width );
            sprtf("\n");
         }
      }
   }
}

void Put_Bit_Array_to_BMP( PGSINFO pgsi, PBYTE pn, int ilat, int ilon, int mx, int my,
                          BYTE colr)
{
   PBYTE pblock = pgsi->bmpinfo.range.pblock;   // extract the BLOCK
   if(pblock) {
      int   x, y, off;
      PBYTE pb;
      BYTE val;
      int width  = pgsi->bmpinfo.range.width;
      // 11 bits high
      for( y = my - 1; y >= 0; y-- )
      {
         pb = &pblock[ (ilat * width) + ilon ];
         // 7 bits wide
         for( x = 0; x < mx; x++ )
         {
            off = (y * mx) + x;
            val = pn[off];
            if(val) {
               // check within BITMAP BOUNDARY
               if( Pt_In_BMP_Boundary( pgsi, ilat, ilon + x ) ) {
                  pb[x] = colr;
               }
            }
         }
         ilat++;  // to next bitmap row
      }
   }
}

void  Add_Text_to_BMP( PGSINFO pgsi, int x, int y, char * txt, int len,
                      BYTE tcolr)
{
   int i;
   int mx, my;
   // ASCII chars are 7 x 11
   Get_Char_Bit_Size( &mx, &my ); // 7 x 11, or whatever
   for( i = 0; i < len; i++ )
   {
      PBYTE pb = Get_Char_Bit_Array( txt[i] );
      Put_Bit_Array_to_BMP( pgsi, pb, y, x, mx, my, tcolr );
      x += 8;
   }
}


void Add_Notice_to_BMP( PGSINFO pgsi )
{
   size_t len;
   char * pn = szTmpBuff;
   time_t ltime;
   char timebuf[26];
   POINT pt;
   SIZE  sz;
   PTSTR pb;
   int   cnt, i;

   Get_Char_Bit_Size( (PINT)&sz.cx, (PINT)&sz.cy );
   // and set initial write position
   pt.y = pgsi->bmpinfo.range.TLilat - 20;
   pt.x = pgsi->bmpinfo.range.TLilon + 10;

   // 1. Put the current limits and scale
   strcpy(pn, "Limits:");
   pb = GetNxtBuf();

   sprintf(pb, "%3.6f", pgsi->bmpinfo.range.n );
   trim_num_IB(pb);
   strcat(pn," n=");
   strcat(pn,pb);
   sprintf(pb, "%3.6f", pgsi->bmpinfo.range.s );
   trim_num_IB(pb);
   strcat(pn,", s=");
   strcat(pn,pb);
   sprintf(pb, "%3.6f", pgsi->bmpinfo.range.e );
   trim_num_IB(pb);
   strcat(pn,", e=");
   strcat(pn,pb);
   sprintf(pb, "%3.6f", pgsi->bmpinfo.range.w );
   trim_num_IB(pb);
   strcat(pn,", w=");
   strcat(pn,pb);
   sprintf(EndBuf(pn),", scale=%d",
      pgsi->bmpinfo.range.scale );

   len = strlen(pn);
   Add_Text_to_BMP( pgsi,
      pt.x, pt.y,
      pn, (int)len,
      pgsi->txt_bmc );
   sprtf( "INFO[%s] at %d,%d"MEOR, pn, pt.x, pt.y );

   cnt = Get_Done_List_Count();
   if( cnt == 1 ) {
      sprintf(pn, "From file [%s], with %ld headers, ",
         pgsi->mv.fn, pgsi->hdrcnt );
      sprintf(EndBuf(pn), "and %ld points.", pgsi->ptcnt );
      len = strlen(pn);
      pt.y -= sz.cy;
      Add_Text_to_BMP( pgsi,
         pt.x, pt.y,
         pn, (int)len,
         pgsi->txt_bmc );
      sprtf( "BMP[%s] at %d,%d"MEOR, pn, pt.x, pt.y );
   } else {
      sprintf(pn, "From %d files:", cnt );
      len = strlen(pn);
      pt.y -= sz.cy;
      Add_Text_to_BMP( pgsi,
         pt.x, pt.y,
         pn, (int)len,
         pgsi->txt_bmc );
      sprtf( "BMP[%s] at %d,%d"MEOR, pn, pt.x, pt.y );

      for( int i = 1; i <= cnt; i++ )
      {
         PTSTR pfn = Get_Done_List_File_Ptr( i );
         if(pfn && *pfn) {
            sprintf(pn, "%d: %s", i, pfn);
            len = strlen(pn);
            pt.y -= sz.cy;
            Add_Text_to_BMP( pgsi,
               pt.x, pt.y,
               pn, (int)len,
               pgsi->txt_bmc );
            sprtf( "BMP[%s] at %d,%d"MEOR, pn, pt.x, pt.y );
         }
      }
   }

   // final time notice ...
   // Get UNIX-style time and display as string. 

   time( &ltime );
   ctime_s(timebuf, 26, &ltime);
   len = 26;
   while(len--) {
      if(timebuf[len] > ' ')
         break;
      timebuf[len] = 0;
   }
   // 12345678901234567890123456
   // Wed Jan 02 02:03:55 1980, with the \n removed
   pb = GetNxtBuf();
   len = 0;
   for( i = 4; i < 11; i++ )
      pb[len++] = timebuf[i]; // Get month, day ...
   for( i = 20; i < 24; i++ )
      pb[len++] = timebuf[i]; // get year
   pb[len++] = ' ';           // add space
   for( i = 11; i < 16; i++ )
      pb[len++] = timebuf[i]; // get hour:minutes
   pb[len] =0;                // zero terminat
   sprintf(pn, "Generated %s, by gshhs_win.exe - Geoff McLane",
      pb);
   len = strlen(pn);

   // set location
   pt.x = pgsi->bmpinfo.range.BRilon - ((long)len * (sz.cx+1));
   pt.y = pgsi->bmpinfo.range.BRilat; // is written above this + sz.cy;

   Add_Text_to_BMP( pgsi,
      pt.x, pt.y,
      pn, (int)len,
      pgsi->txt_bmc );  // BMC_RED );  // color of TEXT
   sprtf( "BMP[%s] at %d,%d"MEOR, pn, pt.x, pt.y );


}

void Add_BMP_Border( PGSINFO pgsi )
{
   PBMPSTUFF pbmps = &pgsi->bmpinfo.bmpstuff;
   PBYTE pblock = pgsi->bmpinfo.range.pblock;   // extract the BLOCK
   BYTE  bdr_colr = pgsi->bdr_bmc;  // extract BORDER color
   if(pblock) {
      PGSRANGE pgsr = &pgsi->bmpinfo.range;
      int   y, x;
      int width  = pgsi->bmpinfo.range.width;
      int height = pgsi->bmpinfo.range.height;
      PBYTE poff1, poff2;
      // lat >= pgsr->s )&&( lat <= pgsr->n )&&( lon >= pgsr->w )&&( lon <= pgsr->e ))
      if( bCheckTLBRVals ) {
         int TLilat, TLilon;
         int BRilat, BRilon;
         Get_BMP_iLatLon( pgsi, pgsr->n, pgsr->w, &TLilat, &TLilon ); // TL
         Get_BMP_iLatLon( pgsi, pgsr->s, pgsr->e, &BRilat, &BRilon ); // BR
         if((TLilat != pgsr->TLilat)||
            (TLilon != pgsr->TLilon)||
            (BRilat != pgsr->BRilat)||
            (BRilon != pgsr->BRilon))
         {
            sprtf( "WARNING: Note TL and BR adjusted ..."MEOR );
            sprtf( "TL was %d,%d, now %d,%d!"MEOR,
               pgsr->TLilat, pgsr->TLilon,
               TLilat, TLilon );
            sprtf( "BR was %d,%d, now %d,%d!"MEOR,
               pgsr->BRilat, pgsr->BRilon,
               BRilat, BRilon );
            sprtf("CHECK ABOVE CHANGE!!!"MEOR );
         }
      }

      if( bAddMinMaxBorder ) {
         if( pgsr->minilat < pgsr->maxilat ) {
            for( y = pgsr->minilat; y <= pgsr->maxilat; y++ )
            {  // vertical lines
               poff1 = &pblock[(y * width) + pgsr->minilon];
               poff2 = &pblock[(y * width) + pgsr->maxilon];
               if( bMinMaxOverwrite ) {
                  *poff1 = bdr_colr; // set BORDER color
                  *poff2 = bdr_colr;
               } else {
                  if( *poff1 == BMC_WHITE )
                     *poff1 = bdr_colr; // set BORDER color
                  if( *poff2 == BMC_WHITE )
                     *poff2 = bdr_colr; // set BORDER color
               }
            }
         }
         if( pgsr->minilon < pgsr->maxilon ) {
            // from left to right lon, set max, min lat
            for( x = pgsr->minilon; x < pgsr->maxilon; x++ )
            {  // horizontal lines
               poff1 = &pblock[(pgsr->minilat * width) + x];
               poff2 = &pblock[(pgsr->maxilat * width) + x];
               if( bMinMaxOverwrite ) {
                  *poff1 = bdr_colr; // set BORDER color
                  *poff2 = bdr_colr;
               } else {
                  if( *poff1 == BMC_WHITE )
                     *poff1 = bdr_colr; // set BORDER color
                  if( *poff2 == BMC_WHITE )
                     *poff2 = bdr_colr; // set BORDER color
               }
            }
         }
      }

      if( bAddLimitsBorder ) {
         for( y = pgsr->BRilat; y < pgsr->TLilat; y++ )
         {  // vertical lines
            poff1 = &pblock[(y * width) + pgsr->TLilon];
            poff2 = &pblock[(y * width) + pgsr->BRilon];
            *poff1 = BMC_BLACK; // set BLACK
            *poff2 = BMC_BLACK; // set BLACK
         }
         // from left to right lon, set max, min lat
         for( x = pgsr->TLilon; x < pgsr->BRilon; x++ )
         {  // horizontal lines
            poff1 = &pblock[(pgsr->TLilat * width) + x];
            poff2 = &pblock[(pgsr->BRilat * width) + x];
            *poff1 = BMC_BLACK;
            *poff2 = BMC_BLACK;
         }
      }
   }
}

BOOL Pt_In_BMP_MINMAX( PGSINFO pgsi, int ilat, int ilon )
{
   PGSRANGE pgsr = &pgsi->bmpinfo.range;
   if( ilat > pgsr->maxilat )
      return FALSE;
   if( ilat < pgsr->minilat )
      return FALSE;
   if( ilon > pgsr->maxilon )
      return FALSE;
   if( ilon < pgsr->minilon )
      return FALSE;

   return TRUE;
}


// check within BITMAP BOUNDARY
BOOL Pt_In_BMP_Boundary( PGSINFO pgsi, int ilat, int ilon )
{
   PGSRANGE pgsr = &pgsi->bmpinfo.range;
   if( ilat > pgsr->TLilat )
      return FALSE;
   if( ilat < pgsr->BRilat )
      return FALSE;
   if( ilon > pgsr->BRilon )
      return FALSE;
   if( ilon < pgsr->TLilon )
      return FALSE;

   return TRUE;
}

BOOL  Pt_In_BMP_Range( PGSINFO pgsi, int ilat, int ilon )
{
   //PBMPSTUFF pbmps = &pgsi->bmpinfo.bmpstuff;
   int width  = pgsi->bmpinfo.range.width;   // pbmps->width;
   int height = pgsi->bmpinfo.range.height;  // pbmps->height;
   if(( ilat >= 0 ) && (ilat < height)&&
      ( ilon >= 0 ) && (ilon < width ))
   {
      return TRUE;
   }
   return FALSE;
}


BOOL  LatLon_In_BMP_Range( PGSINFO pgsi, double lat, double lon )
{
   int ilat, ilon;
   Get_BMP_iLatLon( pgsi, lat, lon, &ilat, &ilon );
   return Pt_In_BMP_Range( pgsi, ilat, ilon );
}

BOOL  LatLon_In_BMP_Boundary( PGSINFO pgsi, double lat, double lon )
{
   int ilat, ilon;
   Get_BMP_iLatLon( pgsi, lat, lon, &ilat, &ilon );
   return Pt_In_BMP_Boundary( pgsi, ilat, ilon );
}

// points are only added, if in BITMAP BOUNDARY
void Add_BMP_LatLon_Pt( PGSINFO pgsi, double lat, double lon, BYTE b,
                       BOOL replace )
{
   PBMPSTUFF pbmps = &pgsi->bmpinfo.bmpstuff;
   PBYTE pblock = pgsi->bmpinfo.range.pblock;   // extract the BLOCK
   if(pblock) {
      int ilat, ilon;
      int width  = pgsi->bmpinfo.range.width;   //pbmps->width;
      int height = pgsi->bmpinfo.range.height;  //pbmps->height;
      Get_BMP_iLatLon( pgsi, lat, lon, &ilat, &ilon );
      // check it is all within BMP range
      if(( ilat >= 0 ) && (ilat < height)&&
         ( ilon >= 0 ) && (ilon < width ))
      {
         if( Pt_In_BMP_Boundary( pgsi, ilat, ilon ) ) {
            PBYTE pb = &pblock[ (ilat * width) + ilon];
            // add em all
            if(( *pb == 0 ) || replace )
               *pb = b;
         }
      } else {
         sprtf( "CHECKME: lat/lon %3.7f,%3.7f, yielded %d,%d, OUTSIDE BLOCK! w=%d, h=%d\n",
            lat, lon,
            ilat, ilon, width, height );
      }
   }
}

// Number bits are ONLY put, if WITHIN BITMAP BOUNDARY
void Put_BMP_Num( PGSINFO pgsi, PBYTE pn, int ilat, int ilon )
{
   //PBMPSTUFF pbmps = &pgsi->bmpinfo.bmpstuff;
   PBYTE pblock = pgsi->bmpinfo.range.pblock;   // extract the BLOCK
   int width  = pgsi->bmpinfo.range.width;   //pbmps->width;
   int height = pgsi->bmpinfo.range.height;  //pbmps->height;
   if( pblock )
   //if(( ilat >= 0 ) && ((ilat+9) < height)&&
   //   ( ilon >= 0 ) && (ilon < width )&&
   //   pblock )
   {
      int   x, y, off;
      PBYTE pb;
      BYTE val;
      BYTE colr = pgsi->txt_bmc;
      // 9 bits high
      for( y = 8; y >= 0; y-- )
      {
         pb = &pblock[ (ilat * width ) + ilon ];
         // 7 bits wide
         for( x = 0; x < 7; x++ )
         {
            off = (y * 7) + x;
            val = pn[off];
            if(val) {
               if( Pt_In_BMP_Boundary( pgsi, ilat, ilon + x ) ) {
                  pb[x] = colr;
               }
            }
         }
         ilat++;  // to next bitmap row
      }
   }
}


// Adjust text to BOUNDARIES, established during POINT accumulation
// but specialized to ONLY handle integer values
void Adjust_BMP_Nums( PGSINFO pgsi, PINT pilat, PINT pilon, BOOL neg, double degs )
{
   PBMPINFO pbmi = &pgsi->bmpinfo;
   PGSRANGE pgsr = &pbmi->range;
   int x, y, xoff;
   int ilat = *pilat;
   int ilon = *pilon;
   int idegs = (int)degs;
   PTSTR pb = GetNxtBuf();
   int len;
   Get_Num_Bit_Size( &x, &y );
   x++;  // bump sizes by 1
   y++;
   len = sprintf(pb,"%d",idegs);
   //xoff = ((neg ? 3 : 2) * x);
   xoff = len * x;
   if( ilat < pgsr->BRilat ) {
      ilat = pgsr->BRilat; // + y;
   }
   if( ilat + y > pgsr->TLilat ) {
      ilat = pgsr->TLilat - y;
   }
   if( ilon < pgsr->TLilon ) {
      ilon = pgsr->TLilon;   // start at LEFT
   }
   if( ilon + xoff > pgsr->BRilon ) {
      ilon = pgsr->BRilon - xoff;
   }
   *pilat = ilat;
   *pilon = ilon;
}

void Adjust_BMP_Text_perminmax( PGSINFO pgsi, PINT pilat, PINT pilon, BOOL neg )
{
   PBMPINFO pbmi = &pgsi->bmpinfo;
   PGSRANGE pgsr = &pbmi->range;
   int x, y, xoff;
   int ilat = *pilat;
   int ilon = *pilon;

   Get_Num_Bit_Size( &x, &y );
   x++;
   y++;
   xoff = ((neg ? 3 : 2) * x);
   if( ilat < pgsr->minilat ) {
      ilat = pgsr->minilat + y;
   }
   if( ilat > pgsr->maxilat ) {
      ilat = pgsr->maxilat;
   }
   if( ilon < pgsr->minilon ) {
      ilon = pgsr->minilon;   // start at LEFT
   }
   if( ilon + xoff > pgsr->maxilon ) {
      ilon = pgsr->maxilon - xoff;
   }
   *pilat = ilat;
   *pilon = ilon;
}


// 7 x 9 bits
void Add_BMP_Lat_Num10( PGSINFO pgsi, double lon )
{
   PBYTE pn;
   double lat;
   int ilat, ilon, inum;
   //for( lat = -80.0; lat <= 80.0; lat += 10.0 )
   for( lat = -90.0; lat <= 90.0; lat += 10.0 )
   {
      if( LatLon_In_BMP_Range( pgsi, lat, lon ) ) {
         BOOL neg = (lat < 0.0 ? TRUE : FALSE);
         Get_BMP_iLatLon( pgsi, lat, lon, &ilat, &ilon );
         ilat -= 4;  // straddle the number over the lat line
         ilon += 4;  // shift it right, off the lon line
         // and maybe SHIFT it to WITHIN the BOUNDARIES as well
         if( gbAdjustText ) Adjust_BMP_Nums( pgsi, &ilat, &ilon, neg, lat );
         //if( lat < 0.0 ) {
         if( neg ) {
            pn = Get_Num_Array( 10 );  // get the MINUS sign
            if(pn) {
               Put_BMP_Num( pgsi, pn, ilat, ilon );
               ilon += 8;
            }
            inum = (int)(-lat / 10);
         } else {
            inum = (int)(lat / 10);
         }
         pn = Get_Num_Array( inum );
         if(pn) {
            Put_BMP_Num( pgsi, pn, ilat, ilon );
            ilon += 8;
         }
         pn = Get_Num_Array( 0 );
         if(pn) {
            Put_BMP_Num( pgsi, pn, ilat, ilon );
            ilon += 8;
         }
      }
   }
}

void Add_BMP_Lon_Num10( PGSINFO pgsi, double lat )
{
   PBYTE pn;
   double lon;
   int ilat, ilon, inum1, inum2;
   //for( lon = -170.0; lon <= 170.0; lon += 10.0 )
   for( lon = -180.0; lon <= 180.0; lon += 10.0 )
   {
      if( LatLon_In_BMP_Range( pgsi, lat, lon ) ) {
         BOOL neg = ((lon < 0.0) ? TRUE : FALSE);
         Get_BMP_iLatLon( pgsi, lat, lon, &ilat, &ilon );
         ilat += 6;  // move above lat line
         ilon -= 10; // straddle the lon line
         // and maybe SHIFT it to WITHIN the BOUNDARIES as well
         if( gbAdjustText ) Adjust_BMP_Nums( pgsi, &ilat, &ilon, neg, lon );
         if( neg ) {
            pn = Get_Num_Array( 10 );  // MINUS SIGN
            if(pn) {
               ilon -= 8;  // back up more for negative numbers
               Put_BMP_Num( pgsi, pn, ilat, ilon );
               ilon += 8;
            }
            inum1 = (int)(-lon / 10);
         } else {
            inum1 = (int)(lon / 10);
         }
         inum2 = inum1 / 10;  // from 17 or 7 = 1 or 0
         inum1 = inum1 % 10;
         if( inum2 ) {
            pn = Get_Num_Array( inum2 );
            if(pn) {
               Put_BMP_Num( pgsi, pn, ilat, ilon );
               ilon += 8;
            }
         }
         pn = Get_Num_Array( inum1 );
         if(pn) {
            Put_BMP_Num( pgsi, pn, ilat, ilon );
            ilon += 8;
         }
         pn = Get_Num_Array( 0 );
         if(pn) {
            Put_BMP_Num( pgsi, pn, ilat, ilon );
            ilon += 8;
         }
      }
   }
}

static double lon10[36+1];
static double lat10[18+1];

void Get_BMP_Grid( PGSINFO pgsi )
{
   PBMPINFO pbmi = &pgsi->bmpinfo;
   PGSRANGE pgsr = &pbmi->range;
   double dwid = pgsr->dwid; // pgsr->e - pgsr->w;  // width, in degrees
   double dhgt = pgsr->dhgt; // pgsr->n - pgsr->s;  // height, in degrees
   double   lat, lon;
   int   latcnt, loncnt, i;
   double dwpart = dwid / 5.0;   // aim for min of 5 lines
   double dhpart = dhgt / 5.0;

   if( !pgsi->bmpinfo.range.pblock )   // do we have a BLOCK
      return;  // nothing to ADD LINES TO ANYWAY!!!

   sprtf( "Center lat %3.8f, lon %3.8f - Lat split %3.7f, Lon split %3.7f ... "MEOR,
      pgsr->centerlat, pgsr->centerlon,
      dhpart, dwpart );

   // any MAJOR 10 degree longitude lines
   loncnt = 0; // store those on BITMAP
   //for( lon = -170.0; lon <= 170.0; lon += 10.0 ) {
   for( lon = -180.0; lon <= 180.0; lon += 10.0 ) {
      if( LatLon_In_BMP_Range( pgsi, pgsr->centerlat, lon ) ) {
         lon10[loncnt++] = lon;
      }
   }
   // add any MAJOR 10 degree latitude lines
   latcnt = 0; // store those on BITMAP
   //for( lat = -80.0; lat <= 80.0; lat += 10.0 ) {
   for( lat = -90.0; lat <= 90.0; lat += 10.0 ) {
      if( LatLon_In_BMP_Range( pgsi, lat, pgsr->centerlon ) ) {
         lat10[latcnt++] = lat;  // store a latitude written
      }
   }
   sprtf( "With TL:%3.6f,%3.6f - BR:%3.6f,%3.6f, Wid=%3.6f, Hgt=%3.6f degrees\n",
      pgsr->n, pgsr->w, pgsr->s, pgsr->e, dwid, dhgt );
   sprtf( "Found %d 10 degrees lat lines, ", latcnt );
   for( i = 0; i < latcnt; i++ ) {
      sprtf( "%3.1f ", lat10[i] );
   }
   sprtf( "\n" );
   sprtf( "Found %d 10 degrees lon lines, ", loncnt );
   for( i = 0; i < loncnt; i++ ) {
      sprtf( "%3.1f ", lon10[i] );
   }
   sprtf( "\n" );

}


int Do_This_lon( PGSINFO pgsi, double lon, double dwpart, int loncnt )
{
   int dolon = 1;
   if(loncnt) {
      double lwrlon = lon - (dwpart / 2.0);
      double uprlon = lon + (dwpart / 2.0);
      double tlon;
      int i;
      for( i = 0; i < loncnt; i++ )
      {
         tlon = lon10[i];
         if(( tlon > lwrlon )&&(tlon < uprlon))
            dolon = 0;
      }
   }
   return dolon;
}

int Do_This_lat( PGSINFO pgsi, double lat, double dhpart, int latcnt )
{
   int dolat = 1;
   if(latcnt) {
      double lwrlat = lat - (dhpart / 2.0);
      double uprlat = lat + (dhpart / 2.0);
      double tlat;
      int i;
      for( i = 0; i < latcnt; i++ )
      {
         tlat = lat10[i];
         if(( tlat > lwrlat )&&(tlat < uprlat))
            dolat = 0;
      }
   }
   return dolat;
}

void Add_BMP_Grid( PGSINFO pgsi )  // like Add_Lat_Lon_Lines( pgsi );
{
   PGSRANGE pgsr = &pgsi->bmpinfo.range;
   PBMPSTUFF pbmps = &pgsi->bmpinfo.bmpstuff;
   PBYTE pblock = pgsi->bmpinfo.range.pblock;   // extract the BLOCK
   char * ptmp = szTmpBuff;
   int mx, my, len;
   POINT pt;
   // ASCII chars are 7 x 11
   Get_Char_Bit_Size( &mx, &my ); // 7 x 11, or whatever
   if(pblock) {
      double dwid = pgsr->dwid;
      double dhgt = pgsr->dhgt; 
      int width  = pgsr->width;  //pbmps->width;    // width of BMP
      int height = pgsr->height; //pbmps->height;   // height of BMP
      double dwpart = dwid / 5.0;
      double dhpart = dhgt / 5.0;
      double lat, lon, x, y;
      int   ilat, ilon;
      int TLilat, TLilon;
      int BRilat, BRilon;
      int   ilatdiff, ilondiff;
      double dlat, dlon, latd, lond;
      double lat2, lon2;
      int loncnt = 0;
      int latcnt = 0;
      int   dolon = 1;
      int   dolat = 1;
      BYTE  ll_colr = pgsi->ll_bmc;
      double min_inc = SG_EPSILON * 100;

      Get_BMP_Grid( pgsi );   // work out WHAT grib lines to PUT
      lat = pgsr->centerlat;  // (pgsr->s + pgsr->n) / 2.0;
      lon = pgsr->centerlon;  // (pgsr->w + pgsr->e) / 2.0;
      Get_BMP_iLatLon( pgsi, lat, lon, &ilat, &ilon );
      ilatdiff = ilat;
      ilondiff = ilon;
      dlat = lat;
      while( ilatdiff == ilat ) {
         dlat += SG_EPSILON;
         Get_BMP_iLatLon( pgsi, dlat, lon, &ilat, &ilon );
      }
      dlon = lon;
      while( ilondiff == ilon ) {
         dlon += SG_EPSILON;
         Get_BMP_iLatLon( pgsi, lat, dlon, &ilat, &ilon );
      }

      latd = dlat - lat;
      lond = dlon - lon;
      //latd = (dlat - lat) * 4.0;
      //lond = (dlon - lon) * 4.0;
      if( latd < min_inc )
         latd = min_inc;
      if( lond < min_inc )
         lond = min_inc;

      Get_BMP_iLatLon( pgsi, pgsr->n, pgsr->w, &TLilat, &TLilon ); // TL
      Get_BMP_iLatLon( pgsi, pgsr->s, pgsr->e, &BRilat, &BRilon ); // BR
      ilatdiff = TLilat - BRilat;
      ilondiff = BRilon - TLilon;

      // use center latitude
      lat2 = (pgsr->s + pgsr->n) / 2.0;
      // add any MAJOR 10 degree longitude lines
      loncnt = 0; // store those on BITMAP
      //for( lon = -170.0; lon <= 170.0; lon += 10.0 )
      for( lon = -180.0; lon <= 180.0; lon += 10.0 )
      {
         if( LatLon_In_BMP_Range( pgsi, lat2, lon ) ) {
            lon10[loncnt++] = lon;  // store a logitude written
            dlon = lon + dwpart;
            if( dlon > pgsr->e )
               dlon = pgsr->e;
            for( lat = pgsr->s; lat <= pgsr->n; lat += dhpart )
            {
               dlat = lat + dhpart;
               if( dlat > pgsr->n )
                  dlat = pgsr->n;
               // vertical lines
               for( y = lat; y <= dlat; y += latd ) {
                  if( LatLon_In_BMP_Range( pgsi, y, lon ) ) {
                     // point is ONLY put, if WITHIN BITMAP BOUNDARY
                     Add_BMP_LatLon_Pt( pgsi, y, lon, ll_colr, FALSE );
                  }
               }
            }
         }
      }
      lon2 = (pgsr->w + pgsr->e) / 2.0;
      // add any MAJOR 10 degree latitude lines
      latcnt = 0; // store those on BITMAP
      //for( lat = -80.0; lat <= 80.0; lat += 10.0 )
      for( lat = -90.0; lat <= 90.0; lat += 10.0 )
      {
         if( LatLon_In_BMP_Range( pgsi, lat, lon2 ) ) {
            lat10[latcnt++] = lat;  // store a latitude written
            dlat = lat + dhpart;
            if( dlat > pgsr->n )
               dlat = pgsr->n;
            for( lon = pgsr->w; lon <= pgsr->e; lon += dwpart )
            {
               dlon = lon + dwpart;
               if( dlon > pgsr->e )
                  dlon = pgsr->e;
               // horizontal lines
               for( x = lon; x <= dlon; x += lond ) {
                  if( LatLon_In_BMP_Range( pgsi, lat, x ) ) {
                     // points are only added, if in BITMAP BOUNDARY
                     Add_BMP_LatLon_Pt( pgsi, lat, x, ll_colr, FALSE );
                  }
               }
            }
         }
      }
      // and their NUMBERS
      lat = (pgsr->s + pgsr->n) / 2.0;
      lon = (pgsr->w + pgsr->e) / 2.0;
      Add_BMP_Lat_Num10( pgsi, lon );
      Add_BMP_Lon_Num10( pgsi, lat );

      // add other latitude / logitude lines
      // skipping any lines close to MAJOR 10 degreee lines
      for( lon = pgsr->w; lon <= pgsr->e; lon += dwpart )
      {
         dlon = lon + dwpart;
         if( dlon > pgsr->e )
            dlon = pgsr->e;
         dolon = Do_This_lon( pgsi, lon, dwpart, loncnt );
         for( lat = pgsr->s; lat <= pgsr->n; lat += dhpart )
         {
            dlat = lat + dhpart;
            if( dlat > pgsr->n )
               dlat = pgsr->n;
            dolat = Do_This_lat( pgsi, lat, dhpart, latcnt );
            // vertical lines
            if(dolon) {
               for( y = lat; y <= dlat; y += latd ) {
                  // points are only added, if in BITMAP BOUNDARY
                  Add_BMP_LatLon_Pt( pgsi, y, lon, ll_colr, FALSE );
               }
            }
            // horizontal lines
            if(dolat) {
               for( x = lon; x <= dlon; x += lond ) {
                  // points are only added, if in BITMAP BOUNDARY
                  Add_BMP_LatLon_Pt( pgsi, lat, x, ll_colr, FALSE );
               }
            }
         }
         if(dolon) {
            // vertical lines - logitudes - add values
            len = sprintf(ptmp,"%3.4f", lon );
            Get_BMP_iLatLon( pgsi, pgsr->centerlat, lon, &ilat, &ilon );
            len = trim_num_IB(ptmp);
            pt.y = ilat + (my / 2); // 8;  // 10;
            pt.x = ilon - (mx * (len / 2)) - mx; // 10;
            if(pt.x < pgsr->TLilon)
               pt.x = pgsr->TLilon;
            Add_Text_to_BMP( pgsi, pt.x, pt.y, ptmp, len, ll_colr ); // or pgsi->txt_bmc
            //sprtf( "Painted %s to %d,%d ..."MEOR, ptmp, pt.x, pt.y );
         }
      }
      for( lat = pgsr->s; lat <= pgsr->n; lat += dhpart )
      {
         dlat = lat + dhpart;
         if( dlat > pgsr->n )
            dlat = pgsr->n;
         dolat = Do_This_lat( pgsi, lat, dhpart, latcnt );
         // horizontal lines
         if(dolat) {
            // hozontal lines - latitudes - add values
            len = sprintf(ptmp,"%3.4f", lat );
            Get_BMP_iLatLon( pgsi, lat, pgsr->centerlon, &ilat, &ilon );
            len = trim_num_IB(ptmp);
            pt.y = ilat - (my / 2); // minus to go DOWN
            pt.x = ilon; // - (mx * (len / 2)) - mx; // 10;
            if((pt.y - my) < pgsr->BRilat)
               pt.y = pgsr->BRilat; // + my;
            //if(pt.x < pgsr->TLilon)
            //   pt.x = pgsr->TLilon;
            Add_Text_to_BMP( pgsi, pt.x, pt.y, ptmp, len, ll_colr ); // or pgsi->txt_bmc
            //sprtf( "Painted %s to %d,%d ..."MEOR, ptmp, pt.x, pt.y );
         }
      }
   }
}

int  Write_User_BMP2( PGSINFO pgsi )
{
   PBMPINFO pbmi = &pgsi->bmpinfo;
   PBMPSTUFF pbmps = &pbmi->bmpstuff;
   PGSRANGE pgsr = &pbmi->range;
   int iret = 0;  // assume FAILED
   char * pfile = pbmi->bmpname;
   PBYTE pblock = pbmi->range.pblock;
   if( pgsi->bmpinfo.bmpname[0] &&
      pbmi->range.pblock &&
      pbmi->range.ptcount )
   {
      // ok. let's party ;=))
      int width  = pbmi->range.width;
      int height = pbmi->range.height;
      int nBPP   = pgsi->bmpinfo.nBPP; // = BMP_DEF_BPP, or as requested;
      FILE * fp = fopen( pfile, "wb" );
      if(fp) {
         int x, y;
         size_t palsz = GetPaletteSize(nBPP);
         BITMAPFILEHEADER * phdr = &_s_hdr;
         LPBITMAPINFOHEADER pbi  = &_s_bi;
         int cols = GetRowWidth( width, nBPP );   // per BPP and rounded to 4 bytes
         int rows = height;
         long bufsize = cols * rows;   // to HOLD the BITS
         size_t dibsz = sizeof(BITMAPINFOHEADER) + palsz + bufsize;
         size_t size  = sizeof(BITMAPFILEHEADER) + dibsz;
         PBYTE pbits;
         PBYTE pout;
         RGBQUAD * ppal = 0; // if ANY
         int row, col;
         BYTE  cb, r, g, b;
         BYTE  cv, ind;
         int nib = 0;

         pbits = (PBYTE)MALLOC( bufsize + palsz );
         if( !pbits ) {
            sprtf( "ERROR: Memory failed on %u bytes ...\n", bufsize );
            fclose(fp);
            goto FAILED;
         }
         ZeroMemory(pbits, bufsize + palsz);
         pbmps->pbits  = pbits;
         pbmps->bms_width  = width;
         pbmps->bms_height = height;
         pbmps->bms_rowwid = cols;
         pbmps->vscale = (double)pbmps->bms_height / (pgsr->n - pgsr->s);
         pbmps->hscale = (double)pbmps->bms_width  / (pgsr->e - pgsr->w);

         if( palsz ) {
            ppal = (RGBQUAD *)( pbits + bufsize );
            // make a PALETTE of COLOURS
         }
         switch( nBPP )
         {
         //case 1:  // monochrome
         //   SetMonochromeMap(ppal);
         //   break;
         case 4:
            SetBPP4Map(ppal);
            break;
         case 8:
            SetBPP8Map(ppal);
            break;
         case 24:
            // nothing to do here - no color map needed
            break;
         default:
            fclose(fp);
            free(pbits);
            sprtf( "WARNING: BPP size of %d NOT HANDLED\n", nBPP );
            goto FAILED;
         }

         // BITMAP SETUP
         // write to BMP
         // ============
         // Bits
         for( y = 0; y < height; y++ )
         {
            row = y;
            pout = &pbits[(row * cols)];  // get ROW pointer
            nib = 0; // FIX20080223 - start the nibble collector for this ROW
            for( x = 0; x < width; x++ )
            {
               cb = pblock[ (y * width) + x ]; // get color
               Get_Pixel_Color( pgsi, cb, &ind, &r, &g, &b, nBPP );
               switch(nBPP)
               {
               case 4:
                  col = (int)(x / 2);   // get COLUMN offset (4-bits)
                  cv = pout[col];   // extract the 2 nibble index
                  if( gbReverseNib ) {
                     if(nib)
                        cv = (ind << 4) | (cv & 0x0f);
                     else
                        cv = (cv & 0xf0) | ind;
                  } else {
                     if(nib)
                        cv = (cv & 0xf0) | ind;
                     else
                        cv = (ind << 4) | (cv & 0x0f);
                  }
                  pout[col] = cv;   // put 2 nibble index
                  // flip the NIB being set
                  if(nib)
                     nib = 0;
                  else
                     nib = 1;
                  break;
               case 8:
                  pout[x] = ind; // NOTE WELL: THE BMC_[color] VALUE USED DIRECTLY
                  break;
               case 24:
                  col = x * 3;   // get COLUMN offset (24-bits)
                  // set COLOR
                  pout[col+2] = r;
                  pout[col+1] = g;
                  pout[col+0] = b;
                  break;
               default:
                  sprtf( "ERROR: UNSUPPORT BITS PER PIXEL VALUE %d!", nBPP );
                  MFREE(pbits);   // free BIT BUFFER, and PALETTE buffer, if any
                  fclose(fp);
                  return 0;
                  break;
               }
            }  // for the WIDTH (x)
         }  // for the HEIGHT (y)

         // Info-header
         InitBitmapInfoHeader( pbi,    // LPBITMAPINFOHEADER lpBmInfoHdr,
                              width,
                              height,
                              nBPP ); // 	  int nBPP )
         // File Header
         phdr->bfType      = DIB_HEADER_MARKER;   // simple "BM" signature
         phdr->bfSize      = (DWORD)size;   // file size
         phdr->bfReserved1 = 0;
         phdr->bfReserved2 = 0;
         phdr->bfOffBits   = (DWORD)sizeof(BITMAPFILEHEADER) + pbi->biSize + palsz;

         // WRITE BITMAP
         fwrite(phdr, 1, sizeof(BITMAPFILEHEADER), fp);
         fwrite(pbi, 1, sizeof(BITMAPINFOHEADER), fp);
         if( ppal && palsz )
            fwrite( ppal, 1, palsz, fp );
         fwrite(pbits, 1, bufsize, fp);

         MFREE(pbits);   // free BIT BUFFER, and PALETTE buffer, if any
         fclose(fp);
         sprtf( "Written [%s] BMP file ... at %d BPP\n", pfile, nBPP );
         iret = 1;
      } else {
   FAILED:
         sprtf( "WARNING: Failed to write %s file ...\n", pfile );
      }
   }
   return iret;

}

// similar to above, except this attempts to cull any white space
// surrounding the bitmap ... thus reducing the BITMAP output
// to only where there is information - that is within the borders
// ===============================================================
int  Write_User_BMP4( PGSINFO pgsi )
{
   PBMPINFO pbmi = &pgsi->bmpinfo;
   PBMPSTUFF pbmps = &pbmi->bmpstuff;
   PGSRANGE pgsr = &pbmi->range;
   int iret = 0;  // assume FAILED
   char * pfile = pbmi->bmpname;
   PBYTE pblock = pbmi->range.pblock;
   if( pgsi->bmpinfo.bmpname[0] &&
      pbmi->range.pblock &&
      pbmi->range.ptcount )
   {
      // ok. let's party ;=))
      int width  = pbmi->range.width;
      int height = pbmi->range.height;
      int nBPP   = pgsi->bmpinfo.nBPP; // = BMP_DEF_BPP, or as requested;
      int x, y, bgnrow, endrow, bgncol, endcol;
      int   bmwidth, bmheight;
      BYTE cb;
      // this is the width and heigth of the bit block generated
      // Bits
      cb = 0;
      for( y = 0; y < height; y++ ) {
         for( x = 0; x < width; x++ ) {
            cb = pblock[ (y * width) + x ]; // get color
            if(cb)
               break;
         }
         if(cb)
            break;
      }
      bgnrow = y; // set the BEGIN row
      cb = 0;
      for( y = height - 1; y >= 0; y-- ) {
         for( x = 0; x < width; x++ ) {
            cb = pblock[ (y * width) + x ]; // get color
            if(cb)
               break;
         }
         if(cb)
            break;
      }
      endrow = y;
      // got the begin and end ROWS
      cb = 0;
      for( x = 0; x < width; x++ ) {
         for( y = 0; y < height; y++ ) {
            cb = pblock[ (y * width) + x ]; // get color
            if(cb)
               break;
         }
         if(cb)
            break;
      }
      bgncol = x;
      cb = 0;
      for( x = width - 1; x >= 0; x-- ) {
         for( y = 0; y < height; y++ ) {
            cb = pblock[ (y * width) + x ]; // get color
            if(cb)
               break;
         }
         if(cb)
            break;
      }
      endcol = x;
      if(( bgnrow < endrow )&&
         ( bgncol < endcol )) {
         // we can proceed
         bmwidth  = endcol - bgncol + 1;
         bmheight = endrow - bgnrow + 1;
      } else {
         sprtf( "WARNING: No data when culled!"MEOR );
         return 0;
      }
      FILE * fp = fopen( pfile, "wb" );
      if(fp) {
         int   bx, by;
         size_t palsz = GetPaletteSize(nBPP);
         BITMAPFILEHEADER * phdr = &_s_hdr;
         LPBITMAPINFOHEADER pbi  = &_s_bi;
         int cols = GetRowWidth( bmwidth, nBPP );   // per BPP and rounded to 4 bytes
         int rows = bmheight;
         long bufsize = cols * rows;   // to HOLD the BITS
         size_t dibsz = sizeof(BITMAPINFOHEADER) + palsz + bufsize;
         size_t size  = sizeof(BITMAPFILEHEADER) + dibsz;
         PBYTE pbits;
         PBYTE pout;
         RGBQUAD * ppal = 0; // if ANY
         int row, col;
         BYTE  r, g, b;
         BYTE  cv, ind;
         int nib = 0;

         pbits = (PBYTE)MALLOC( bufsize + palsz );
         if( !pbits ) {
            sprtf( "ERROR: Memory failed on %u bytes ...\n", bufsize );
            fclose(fp);
            goto FAILED;
         }
         ZeroMemory(pbits, bufsize + palsz);
         pbmps->pbits  = pbits;
         pbmps->bms_width  = bmwidth;
         pbmps->bms_height = bmheight;
         pbmps->bms_rowwid = cols;
         pbmps->vscale = (double)pbmps->bms_height / (pgsr->n - pgsr->s);
         pbmps->hscale = (double)pbmps->bms_width  / (pgsr->e - pgsr->w);

         if( palsz ) {
            ppal = (RGBQUAD *)( pbits + bufsize );
            // make a PALETTE of COLOURS
         }
         switch( nBPP )
         {
         //case 1:  // monochrome
         //   SetMonochromeMap(ppal);
         //   break;
         case 4:
            SetBPP4Map(ppal);
            break;
         case 8:
            SetBPP8Map(ppal);
            break;
         case 24:
            // nothing to do here - no color map needed
            break;
         default:
            fclose(fp);
            free(pbits);
            sprtf( "WARNING: BPP size of %d NOT HANDLED\n", nBPP );
            goto FAILED;
         }

         // BITMAP SETUP
         // write to BMP
         // ============
         // Bits
         for( y = bgnrow, by = 0; y <= endrow; y++, by++ )
         {
            row = by;
            pout = &pbits[(row * cols)];  // get ROW pointer
            nib = 0; // FIX20080223 - start the nibble collector for this ROW
            for( x = bgncol, bx = 0; x <= endcol; x++, bx++ )
            {
               cb = pblock[ (y * width) + x ]; // get color
               Get_Pixel_Color( pgsi, cb, &ind, &r, &g, &b, nBPP );
               switch(nBPP)
               {
               case 4:
                  col = (int)(bx / 2);   // get COLUMN offset (4-bits)
                  cv = pout[col];   // extract the 2 nibble index
                  if( gbReverseNib ) {
                     if(nib)
                        cv = (ind << 4) | (cv & 0x0f);
                     else
                        cv = (cv & 0xf0) | ind;
                  } else {
                     if(nib)
                        cv = (cv & 0xf0) | ind;
                     else
                        cv = (ind << 4) | (cv & 0x0f);
                  }
                  pout[col] = cv;   // put 2 nibble index
                  // flip the NIB being set
                  if(nib)
                     nib = 0;
                  else
                     nib = 1;
                  break;
               case 8:
                  pout[bx] = ind; // NOTE WELL: THE BMC_[color] VALUE USED DIRECTLY
                  break;
               case 24:
                  col = bx * 3;   // get COLUMN offset (24-bits)
                  // set COLOR
                  pout[col+2] = r;
                  pout[col+1] = g;
                  pout[col+0] = b;
                  break;
               default:
                  sprtf( "ERROR: UNSUPPORT BITS PER PIXEL VALUE %d!", nBPP );
                  MFREE(pbits);   // free BIT BUFFER, and PALETTE buffer, if any
                  fclose(fp);
                  return 0;
                  break;
               }
            }  // for the WIDTH (x)
         }  // for the HEIGHT (y)

         // Info-header
         InitBitmapInfoHeader( pbi,    // LPBITMAPINFOHEADER lpBmInfoHdr,
                              bmwidth,
                              bmheight,
                              nBPP ); // 	  int nBPP )
         // File Header
         phdr->bfType      = DIB_HEADER_MARKER;   // simple "BM" signature
         phdr->bfSize      = (DWORD)size;   // file size
         phdr->bfReserved1 = 0;
         phdr->bfReserved2 = 0;
         phdr->bfOffBits   = (DWORD)sizeof(BITMAPFILEHEADER) + pbi->biSize + palsz;

         // WRITE BITMAP
         fwrite(phdr, 1, sizeof(BITMAPFILEHEADER), fp);
         fwrite(pbi, 1, sizeof(BITMAPINFOHEADER), fp);
         if( ppal && palsz )
            fwrite( ppal, 1, palsz, fp );
         fwrite(pbits, 1, bufsize, fp);

         MFREE(pbits);   // free BIT BUFFER, and PALETTE buffer, if any
         fclose(fp);
         sprtf( "Written [%s] BMP file ... at %d BPP\n", pfile, nBPP );
         iret = 1;
      } else {
   FAILED:
         sprtf( "WARNING: Failed to write %s file ...\n", pfile );
      }
   }
   return iret;

}


int  Write_Block_BMP( PGSINFO pgsi )
{
   PTSTR pfile = "tempblock2.bmp";
   PBMPINFO pbmi = &pgsi->bmpinfo;
   int width  = pbmi->range.width;
   int height = pbmi->range.height;
   int nBPP   = 8;   // pgsi->bmpinfo.nBPP; // = BMP_DEF_BPP, or as requested;
   BITMAPFILEHEADER * phdr = &_s_hdr;
   LPBITMAPINFOHEADER pbi  = &_s_bi;
   int cols = GetRowWidth( width, nBPP );   // per BPP and rounded to 4 bytes
   int rows = height;
   long bufsize = cols * rows;   // to HOLD the BITS
   size_t palsz = GetPaletteSize(nBPP);
   size_t dibsz = sizeof(BITMAPINFOHEADER) + palsz + bufsize;
   size_t size  = sizeof(BITMAPFILEHEADER) + dibsz;
   RGBQUAD *ppal = (RGBQUAD *)Get_Std_Map();
   PBYTE pblock = pgsi->bmpinfo.range.pblock;
   PBYTE pbits = (PBYTE)MALLOC(bufsize);
   FILE * fp;
   PBYTE pout;
   int   x, y;
   BYTE  cb;
   CHKMEM(pbits,bufsize);
   ZeroMemory(pbits,bufsize);
   fp = fopen( pfile, "wb" );
   if(!fp) {
      MFREE(pbits);
      return 0;
   }

   if(ppal && palsz) Set_BMP_8BPP(ppal);
   // move the BITS
   for( y = 0; y < height; y++ ) {
      pout = &pbits[(y * cols)];  // get ROW pointer
      for( x = 0; x < width; x++ ) {
         cb = pblock[ (y * width) + x ]; // get color
         pout[x] = cb;
      }
   }

   InitBitmapInfoHeader( pbi,    // LPBITMAPINFOHEADER lpBmInfoHdr,
      width, height, nBPP );
   
   // File Header
   phdr->bfType      = DIB_HEADER_MARKER;   // simple "BM" signature
   phdr->bfSize      = (DWORD)size;   // file size
   phdr->bfReserved1 = 0;
   phdr->bfReserved2 = 0;
   phdr->bfOffBits   = (DWORD)sizeof(BITMAPFILEHEADER) + pbi->biSize + palsz;

   // WRITE BITMAP
   fwrite(phdr, 1, sizeof(BITMAPFILEHEADER), fp);
   fwrite(pbi, 1, sizeof(BITMAPINFOHEADER), fp);
   if( ppal && palsz )
      fwrite( ppal, 1, palsz, fp );
   fwrite(pbits, 1, bufsize, fp);
   fclose(fp);
   MFREE(pbits);
   sprtf( "Written %s BMP file ... at 8 BPP\n", pfile );
   return 1;
}


int  Write_User_BMP( PGSINFO pgsi )
{
   PBMPINFO pbmi = &pgsi->bmpinfo;
   PBMPSTUFF pbmps = &pbmi->bmpstuff;
   int iret = 0;  // assume FAILED
   if( pgsi->bmpinfo.bmpname[0] &&
      pbmi->range.pblock &&
      pbmi->range.ptcount )
   {
      pbmps->bms_width  = pbmi->range.width;
      pbmps->bms_height = pbmi->range.height;
      Add_Notice_to_BMP( pgsi );
      if( g_AddBorder )
         Add_BMP_Border( pgsi ); // like Add_Min_Max_Border( pgsi );
      if( g_AddGrid )
         Add_BMP_Grid( pgsi );   // like Add_Lat_Lon_Lines( pgsi );

      // and WRITE USER BITMAP
      iret = 0;
      if( g_bCullBitmap ) {
         iret = Write_User_BMP4( pgsi );
      }
      if( !iret )
         iret = Write_User_BMP2( pgsi );

      if( bWriteBlockBMP )
         Write_Block_BMP( pgsi );

   }
   return iret;
}



// eof - gshhs_bmp.c
