/*	$Id: gshhs.c,v 1.18 2008/01/23 03:22:49 guru Exp $
 *
 *	Copyright (c) 1996-2008 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 * PROGRAM:	gshhs.c
 * AUTHOR:	Paul Wessel (pwessel@hawaii.edu)
 * CREATED:	JAN. 28, 1996
 * PURPOSE:	To extract ASCII data from binary shoreline data
 *		as described in the 1996 Wessel & Smith JGR Data Analysis Note.
 * VERSION:	1.1 (Byte flipping added)
 *		1.2 18-MAY-1999:
 *		   Explicit binary open for DOS systems
 *		   POSIX.1 compliant
 *		1.3 08-NOV-1999: Released under GNU GPL
 *		1.4 05-SEPT-2000: Made a GMT supplement; FLIP no longer needed
 *		1.5 14-SEPT-2004: Updated to deal with latest GSHHS database (1.3)
 *		1.6 02-MAY-2006: Updated to deal with latest GSHHS database (1.4)
 *		1.7 11-NOV-2006: Fixed bug in computing level (&& vs &)
 *		1.8 31-MAR-2007: Updated to deal with latest GSHHS database (1.5)
 *		1.9 27-AUG-2007: Handle line data as well as polygon data
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: www.soest.hawaii.edu/pwessel */

#if (defined(_MSC_VER) && (_MSC_VER > 1300))
#define _CRT_SECURE_NO_WARNINGS
#endif

#undef SPCL_DEBUG_EXTRA
#define SOME_DEBUG

#include "gshhs.h"

struct GSHHS h2;
struct POINT p2;

int gshhs_main (int argc, char **argv)
{
	double w, e, s, n, area, lon, lat;
	char source, kind[2] = {'P', 'L'}, *name[2] = {"polygon", "line"};
	FILE	*fp;
	int	k, line, max_east = 270000000, info, flip, level, version, greenwich, src;
   size_t n_read;
	struct	POINT p;
	struct GSHHS h;
        
	if (argc < 2 || argc > 3) {
		fprintf (stderr, "gshhs v. %s ASCII export tool\n", GSHHS_PROG_VERSION);
		fprintf (stderr, "usage:  gshhs gshhs_[f|h|i|l|c].b [-L] > ascii.dat\n");
		fprintf (stderr, "-L will only list headers (no data output)\n");
		exit (EXIT_FAILURE);
	}

	info = (argc == 3);
	if ((fp = fopen (argv[1], "rb")) == NULL ) {
		fprintf (stderr, "gshhs:  Could not find file %s.\n", argv[1]);
		exit (EXIT_FAILURE);
   } else {
#if (defined(SPCL_DEBUG_EXTRA) || defined(SOME_DEBUG))
      printf( "Processing file [%s], using GSHHS structure size = %d bytes... \n",
         argv[1],
         sizeof(struct GSHHS) );
#endif
   }
		
	n_read = fread ((void *)&h, (size_t)sizeof (struct GSHHS), (size_t)1, fp);
   h2 = h;
	version = (h.flag >> 8) & 255;
	flip = (version != GSHHS_DATA_VERSION);	/* Take as sign that byte-swabbing is needed */

	while (n_read == 1) {
      h2 = h;  // copy the RAW READ
		if (flip) {
			h.id = swabi4 ((unsigned int)h.id);
			h.n  = swabi4 ((unsigned int)h.n);
			h.west  = swabi4 ((unsigned int)h.west);
			h.east  = swabi4 ((unsigned int)h.east);
			h.south = swabi4 ((unsigned int)h.south);
			h.north = swabi4 ((unsigned int)h.north);
			h.area  = swabi4 ((unsigned int)h.area);
			h.flag  = swabi4 ((unsigned int)h.flag);
		}
		level = h.flag & 255;
		version = (h.flag >> 8) & 255;
		greenwich = (h.flag >> 16) & 255;
		src = (h.flag >> 24) & 255;
		w = h.west  * GSHHS_SCL;	/* Convert from microdegrees to degrees */
		e = h.east  * GSHHS_SCL;
		s = h.south * GSHHS_SCL;
		n = h.north * GSHHS_SCL;
		source = (src == 1) ? 'W' : 'C';	/* Either WVS or CIA (WDBII) pedigree */
		line = (h.area) ? 0 : 1;		/* Either Polygon (0) or Line (1) (if no area) */
		area = 0.1 * h.area;			/* Now im km^2 */
#ifdef   SPCL_DEBUG_EXTRA
      printf ("type  : %s\n", name[line] );
      printf ("id    : %6d (%d) [%#x]\n", h.id, h2.id, h2.id );
      printf ("num   : %8d (%d) [%#x]\n", h.n, h2.n, h2.n );
      printf ("level : %2d (%d) [%#x]\n", level, (h2.flag & 255), (h2.flag & 255));
      printf ("source: %2c (%d) [%#x]\n", source, (h2.flag >> 24) & 255, (h2.flag >> 24) & 255);
      printf ("area  : %13.3f (%f) [%#x]\n", area, h2.area, h2.area);
      printf ("w     : %10.5f (%f) [%#x]\n", w, h2.west, h2.west );
      printf ("e     : %10.5f (%f) [%#x]\n", e, h2.east, h2.east );
      printf ("s     : %10.5f (%f) [%#x]\n", s, h2.south, h2.south );
      printf ("n     : %10.5f (%f) [%#x]\n", n, h2.north, h2.north );
#else // !SPCL_DEBUG_EXTRA
#ifdef SOME_DEBUG
      printf ("type  : %s\n", name[line] );
      printf ("id    : %6d\n", h.id );
      printf ("num   : %8d (%#x)as(%#x)\n", h.n, h.n, h2.n );
      printf ("level : %2d\n", level );
      printf ("source: %2c\n", source );
      printf ("area  : %13.3f km^2 (%#x)as(%#x)\n", area, h.area, h2.area );
      printf ("w     : %10.5f (%#x)as(%#x)\n", w, h.west, h2.west );
      printf ("e     : %10.5f (%#x)as(%#x)\n", e, h.east, h2.east );
      printf ("s     : %10.5f (%#x)as(%#x)\n", s, h.south, h2.south );
      printf ("n     : %10.5f (%#x)as(%#x)\n", n, h.north, h2.north );
#else
		printf ("%c %6d%8d%2d%2c%13.3f%10.5f%10.5f%10.5f%10.5f\n", kind[line], h.id, h.n, level, source, area, w, e, s, n);
#endif
#endif // SPCL_DEBUG_EXTRA y/n

		if (info) {	/* Skip data, only want headers */
			fseek (fp, (long)(h.n * sizeof(struct POINT)), SEEK_CUR);
		}
		else {
			for (k = 0; k < h.n; k++) {

				if (fread ((void *)&p, (size_t)sizeof(struct POINT), (size_t)1, fp) != 1) {
					fprintf (stderr, "gshhs:  Error reading file %s for %s %d, point %d.\n", argv[1], name[line], h.id, k);
					exit (EXIT_FAILURE);
				}
            p2 = p;
				if (flip) {
					p.x = swabi4 ((unsigned int)p.x);
					p.y = swabi4 ((unsigned int)p.y);
				}
				lon = p.x * GSHHS_SCL;
				if (greenwich && p.x > max_east) lon -= 360.0;
				lat = p.y * GSHHS_SCL;
#ifdef   SPCL_DEBUG_EXTRA
            printf (" lat : %10.5f (%f), lon : %10.5f (%f)\n", lon, p2.x, lat, p2.y);
#else // !#ifdef   SPCL_DEBUG_EXTRA
#ifdef SOME_DEBUG
            printf ("%4d: lat : %10.5f, lon : %10.5f\n", (k+1), lon, lat);
#else
            printf ("%10.5f%10.5f\n", lon, lat);
#endif
#endif // #ifdef   SPCL_DEBUG_EXTRA y/n
			}
		}
		max_east = 180000000;	/* Only Eurasiafrica needs 270 */
		n_read = fread((void *)&h, (size_t)sizeof (struct GSHHS), (size_t)1, fp);
      h2 = h;
	}
		
	fclose (fp);

	exit (EXIT_SUCCESS);
}

struct GSHHS13 {	/* Global Self-consistent Hierarchical High-resolution Shorelines */
	int id;				/* Unique polygon id number, starting at 0 */
	int n;				/* Number of points in this polygon */
	int level;			/* 1 land, 2 lake, 3 island_in_lake, 4 pond_in_island_in_lake */
	int west, east, south, north;	/* min/max extent in micro-degrees */
	int area;			/* Area of polygon in 1/10 km^2 */
	int version;			/* Version of GSHHS polygon (3 is latest and first with this item) */
	short int greenwich;		/* Greenwich is 1 if Greenwich is crossed */
	short int source;		/* 0 = CIA WDBII, 1 = WVS */
};

#define swabi2(i2) (((i2) >> 8) + (((i2) & 255) << 8))

struct GSHHS13 h3;

int gshhs_main13 (int argc, char **argv)
{
	double w, e, s, n, area, lon, lat;
	char source, kind[2] = {'P', 'L'}, *name[2] = {"polygon", "line"};
	FILE	*fp;
	int	k, line, max_east = 270000000, info, flip, level, version, greenwich, src;
   size_t n_read;
	struct	POINT p;
	struct GSHHS13 h;
        
	if (argc < 2 || argc > 3) {
		fprintf (stderr, "gshhs v. %s ASCII export tool\n", GSHHS_PROG_VERSION);
		fprintf (stderr, "usage:  gshhs gshhs_[f|h|i|l|c].b [-L] > ascii.dat\n");
		fprintf (stderr, "-L will only list headers (no data output)\n");
		exit (EXIT_FAILURE);
	}

	info = (argc == 3);
	if ((fp = fopen (argv[1], "rb")) == NULL ) {
		fprintf (stderr, "gshhs:  Could not find file %s.\n", argv[1]);
		exit (EXIT_FAILURE);
   } else {
#if (defined(SPCL_DEBUG_EXTRA) || defined(SOME_DEBUG))
      printf( "Processing file [%s], using GSHHS13 structure size = %d bytes... \n",
         argv[1],
         sizeof(struct GSHHS13) );
#endif
   }
		
	n_read = fread ((void *)&h, (size_t)sizeof (struct GSHHS13), (size_t)1, fp);
   h3 = h;
	//version = (h.flag >> 8) & 255;
   version = h.version;
	flip = (version != GSHHS_DATA_VERSION);	/* Take as sign that byte-swabbing is needed */

	while (n_read == 1) {
      h3 = h;  // copy the RAW READ
		if (flip) {
			h.id = swabi4 ((unsigned int)h.id);
			h.n  = swabi4 ((unsigned int)h.n);
			h.west  = swabi4 ((unsigned int)h.west);
			h.east  = swabi4 ((unsigned int)h.east);
			h.south = swabi4 ((unsigned int)h.south);
			h.north = swabi4 ((unsigned int)h.north);
			h.area  = swabi4 ((unsigned int)h.area);
			//h.flag  = swabi4 ((unsigned int)h.flag);
         h.version = swabi4( h.version );			/* Version of GSHHS polygon (3 is latest and first with this item) */
         h.greenwich = swabi2( h.greenwich );   /* Greenwich is 1 if Greenwich is crossed */
         h.source = swabi2( h.source );   /* 0 = CIA WDBII, 1 = WVS */
		}
		level = h.level;
		version = h.version;
		greenwich = h.greenwich;
		src = h.source;
		w = h.west  * GSHHS_SCL;	/* Convert from microdegrees to degrees */
		e = h.east  * GSHHS_SCL;
		s = h.south * GSHHS_SCL;
		n = h.north * GSHHS_SCL;
		source = (src == 1) ? 'W' : 'C';	/* Either WVS or CIA (WDBII) pedigree */
		line = (h.area) ? 0 : 1;		/* Either Polygon (0) or Line (1) (if no area) */
		area = 0.1 * h.area;			/* Now im km^2 */
#ifdef   SPCL_DEBUG_EXTRA
      printf ("type  : %s\n", name[line] );
      printf ("id    : %6d (%d) [%#x]\n", h.id, h2.id, h2.id );
      printf ("num   : %8d (%d) [%#x]\n", h.n, h2.n, h2.n );
      printf ("level : %2d (%d) [%#x]\n", level, (h2.flag & 255), (h2.flag & 255));
      printf ("source: %2c (%d) [%#x]\n", source, (h2.flag >> 24) & 255, (h2.flag >> 24) & 255);
      printf ("area  : %13.3f (%f) [%#x]\n", area, h2.area, h2.area);
      printf ("w     : %10.5f (%f) [%#x]\n", w, h2.west, h2.west );
      printf ("e     : %10.5f (%f) [%#x]\n", e, h2.east, h2.east );
      printf ("s     : %10.5f (%f) [%#x]\n", s, h2.south, h2.south );
      printf ("n     : %10.5f (%f) [%#x]\n", n, h2.north, h2.north );
#else // !SPCL_DEBUG_EXTRA
#ifdef SOME_DEBUG
      printf ("type  : %s\n", name[line] );
      printf ("id    : %6d\n", h.id );
      printf ("num   : %8d (%#x)as(%#x)\n", h.n, h.n, h3.n );
      printf ("level : %2d\n", level );
      printf ("source: %2c\n", source );
      printf ("area  : %13.3f km^2 (%#x)as(%#x)\n", area, h.area, h3.area );
      printf ("w     : %10.5f (%#x)as(%#x)\n", w, h.west, h3.west );
      printf ("e     : %10.5f (%#x)as(%#x)\n", e, h.east, h3.east );
      printf ("s     : %10.5f (%#x)as(%#x)\n", s, h.south, h3.south );
      printf ("n     : %10.5f (%#x)as(%#x)\n", n, h.north, h3.north );
#else
		printf ("%c %6d%8d%2d%2c%13.3f%10.5f%10.5f%10.5f%10.5f\n", kind[line], h.id, h.n, level, source, area, w, e, s, n);
#endif
#endif // SPCL_DEBUG_EXTRA y/n

		if (info) {	/* Skip data, only want headers */
			fseek (fp, (long)(h.n * sizeof(struct POINT)), SEEK_CUR);
		}
		else {
			for (k = 0; k < h.n; k++) {

				if (fread ((void *)&p, (size_t)sizeof(struct POINT), (size_t)1, fp) != 1) {
					fprintf (stderr, "gshhs:  Error reading file %s for %s %d, point %d.\n", argv[1], name[line], h.id, k);
					exit (EXIT_FAILURE);
				}
            p2 = p;
				if (flip) {
					p.x = swabi4 ((unsigned int)p.x);
					p.y = swabi4 ((unsigned int)p.y);
				}
				lon = p.x * GSHHS_SCL;
				if (greenwich && p.x > max_east) lon -= 360.0;
				lat = p.y * GSHHS_SCL;
#ifdef   SPCL_DEBUG_EXTRA
            printf (" lat : %10.5f (%f), lon : %10.5f (%f)\n", lon, p2.x, lat, p2.y);
#else // !#ifdef   SPCL_DEBUG_EXTRA
#ifdef SOME_DEBUG
            printf ("%4d: lat : %10.5f, lon : %10.5f\n", (k+1), lon, lat);
#else
            printf ("%10.5f%10.5f\n", lon, lat);
#endif
#endif // #ifdef   SPCL_DEBUG_EXTRA y/n
			}
		}
		max_east = 180000000;	/* Only Eurasiafrica needs 270 */
		n_read = fread((void *)&h, (size_t)sizeof (struct GSHHS), (size_t)1, fp);
      h3 = h;
	}
		
	fclose (fp);

	exit (EXIT_SUCCESS);
}


int gshhs_main_ORG (int argc, char **argv)
{
	double w, e, s, n, area, lon, lat;
	char source, kind[2] = {'P', 'L'}, *name[2] = {"polygon", "line"};
	FILE	*fp;
	int	k, line, max_east = 270000000, info, n_read, flip, level, version, greenwich, src;
	struct	POINT p;
	struct GSHHS h;
        
	if (argc < 2 || argc > 3) {
		fprintf (stderr, "gshhs v. %s ASCII export tool\n", GSHHS_PROG_VERSION);
		fprintf (stderr, "usage:  gshhs gshhs_[f|h|i|l|c].b [-L] > ascii.dat\n");
		fprintf (stderr, "-L will only list headers (no data output)\n");
		exit (EXIT_FAILURE);
	}

	info = (argc == 3);
	if ((fp = fopen (argv[1], "rb")) == NULL ) {
		fprintf (stderr, "gshhs:  Could not find file %s.\n", argv[1]);
		exit (EXIT_FAILURE);
	}
		
	n_read = (int)fread ((void *)&h, (size_t)sizeof (struct GSHHS), (size_t)1, fp);
	version = (h.flag >> 8) & 255;
	flip = (version != GSHHS_DATA_VERSION);	/* Take as sign that byte-swabbing is needed */
	
	while (n_read == 1) {
		if (flip) {
			h.id = swabi4 ((unsigned int)h.id);
			h.n  = swabi4 ((unsigned int)h.n);
			h.west  = swabi4 ((unsigned int)h.west);
			h.east  = swabi4 ((unsigned int)h.east);
			h.south = swabi4 ((unsigned int)h.south);
			h.north = swabi4 ((unsigned int)h.north);
			h.area  = swabi4 ((unsigned int)h.area);
			h.flag  = swabi4 ((unsigned int)h.flag);
		}
		level = h.flag & 255;
		version = (h.flag >> 8) & 255;
		greenwich = (h.flag >> 16) & 255;
		src = (h.flag >> 24) & 255;
		w = h.west  * GSHHS_SCL;	/* Convert from microdegrees to degrees */
		e = h.east  * GSHHS_SCL;
		s = h.south * GSHHS_SCL;
		n = h.north * GSHHS_SCL;
		source = (src == 1) ? 'W' : 'C';	/* Either WVS or CIA (WDBII) pedigree */
		line = (h.area) ? 0 : 1;		/* Either Polygon (0) or Line (1) (if no area) */
		area = 0.1 * h.area;			/* Now im km^2 */

		printf ("%c %6d%8d%2d%2c%13.3f%10.5f%10.5f%10.5f%10.5f\n", kind[line], h.id, h.n, level, source, area, w, e, s, n);

		if (info) {	/* Skip data, only want headers */
			fseek (fp, (long)(h.n * sizeof(struct POINT)), SEEK_CUR);
		}
		else {
			for (k = 0; k < h.n; k++) {

				if (fread ((void *)&p, (size_t)sizeof(struct POINT), (size_t)1, fp) != 1) {
					fprintf (stderr, "gshhs:  Error reading file %s for %s %d, point %d.\n", argv[1], name[line], h.id, k);
					exit (EXIT_FAILURE);
				}
				if (flip) {
					p.x = swabi4 ((unsigned int)p.x);
					p.y = swabi4 ((unsigned int)p.y);
				}
				lon = p.x * GSHHS_SCL;
				if (greenwich && p.x > max_east) lon -= 360.0;
				lat = p.y * GSHHS_SCL;
				printf ("%10.5f%10.5f\n", lon, lat);
			}
		}
		max_east = 180000000;	/* Only Eurasiafrica needs 270 */
		n_read = (int)fread((void *)&h, (size_t)sizeof (struct GSHHS), (size_t)1, fp);
	}
		
	fclose (fp);

	exit (EXIT_SUCCESS);
}

#define  DEF_IN_FILE    "..\\data_1.6\\gshhs_i.b"
//#define  DEF_IN_FILE    "..\\data_1.3\\gshhs_i.b"
//#define  DEF_IN_FILE    "..\\gshhs_isl\\gshhs_isle.shx"
//#define DEF_IN_FILE "..\\GMT\\share\\coast\\binned_border_c.cdf"
char * def_file = DEF_IN_FILE;
char * def_args[3];

#define  SWAPIT(a)   a = swabi4(a)

char header_block[68];

struct GSHHS gs1;
struct GSHHS gs2;

int gshhs_main2 (int argc, char **argv)
{
	FILE	*fp;
   char * fn = argv[1];
   size_t n_read;
	struct	POINT p;
	struct	POINT p2;
	struct	POINT p3;
   double lat, lon, flat, flon;
   char * tmp1 = &header_block[ (sizeof(struct GSHHS13)) ];
   char * tmp2 = &header_block[ (sizeof(struct GSHHS)) ];
   struct GSHHS13 * pgs13 = (struct GSHHS13 *)&header_block[0];
   struct GSHHS13 * pgs132 = (struct GSHHS13 *)tmp1;
   struct GSHHS * pgs15 = (struct GSHHS *)&header_block[0];
   struct GSHHS * pgs152 = (struct GSHHS *)tmp2;
   struct GSHHS * pg1 = &gs1;
   struct GSHHS * pg2 = &gs2;

   size_t bcnt = 0;
   size_t fbcnt;
   size_t hbc, fhbc;
   printf( "Size of GSHHS 1.3 structure = %d bytes, 1.5 %d bytes\n", sizeof(struct GSHHS13),
      sizeof(struct GSHHS) );
   printf( "Loading file %s ...\n", fn );
	if ((fp = fopen (fn, "rb")) == NULL ) {
		fprintf (stderr, "gshhs:  Could not find file %s.\n", fn);
		exit (EXIT_FAILURE);
	}
	n_read = fread ((void *)&header_block, (size_t)sizeof(header_block), (size_t)1, fp);
   gs1 = *pgs15;
   SWAPIT(pg1->id);
   SWAPIT(pg1->n);   /* Number of points in this polygon */
   SWAPIT(pg1->flag);
   SWAPIT(pg1->west);
   SWAPIT(pg1->east);
   SWAPIT(pg1->south);
   SWAPIT(pg1->north);	/* min/max extent in micro-degrees */
   gs2 = *pgs152;
   SWAPIT(pg2->id);
   SWAPIT(pg2->n);   /* Number of points in this polygon */
   SWAPIT(pg2->flag);
   SWAPIT(pg2->west);
   SWAPIT(pg2->east);
   SWAPIT(pg2->south);
   SWAPIT(pg2->north);	/* min/max extent in micro-degrees */

	while(n_read == 1) {
			if (fread ((void *)&p, (size_t)sizeof(struct POINT), (size_t)1, fp) != 1) {
            fbcnt = swabi4(bcnt);
            hbc = bcnt / 2;
            fhbc = swabi4(hbc);
            printf ( "gshhs:  END OF FILE, %d (%#x) blocks read flip %#x ...\n", bcnt, bcnt,
               fbcnt );
            printf ( "gshhs:  END OF FILE, %d (%#x) blocks2 read flip %#x ...\n", hbc, hbc,
               fhbc );
            break;
			}
			p2.x = swabi4 ((unsigned int)p.x);
			p2.y = swabi4 ((unsigned int)p.y);
         p3 = p;
			flon = p2.x * GSHHS_SCL;
			//if (greenwich && p.x > max_east) lon -= 360.0;
			flat = p2.y * GSHHS_SCL;
			lon = p.x * GSHHS_SCL;
			//if (greenwich && p.x > max_east) lon -= 360.0;
			lat = p.y * GSHHS_SCL;
			//printf ("lon %10.5f, lat %10.5f (lon %10.5f, lat %10.5f)\n", flon, flat, lon, lat);
			printf ("lon %f, lat %f (lon %f, lat %f)\n", flon, flat, lon, lat);

         bcnt++;
   }

   return 0;
}

extern void Process_File( char * fn );

int main (int argc, char **argv)
{
   def_args[0] = argv[0];
   def_args[1] = def_file;
   def_args[2] = 0;
   Process_File( def_file );
   //return gshhs_main13 (2, def_args);
   //return gshhs_main (2, def_args);
   //return gshhs_main2 (2, def_args);
   return 0;
}

// eof 
