//
// "$Id: Fl.H 6439 2008-10-15 13:46:06Z fabien $"
//
// Main header file for the Fast Light Tool Kit (FLTK).
//
// Copyright 1998-2008 by Bill Spitzak and others.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.
//
// Please report all bugs and problems on the following page:
//
//     http://www.fltk.org/str.php
//

/* \file  
    Fl static class .
 */

#ifndef Fl_H
#  define Fl_H

#ifdef HAVE_CAIRO
# include <FL/Fl_Cairo.H>
#endif

#  include "fl_utf8.h"
#  include "Enumerations.H"
#  ifndef Fl_Object
#    define Fl_Object Fl_Widget	/**< for back compatibility - use Fl_Widget! */
#  endif

#  ifdef check
#    undef check
#  endif


class Fl_Widget;
class Fl_Window;
class Fl_Image;
struct Fl_Label;

/** signature of some label drawing functions passed as parameters */
typedef void (Fl_Label_Draw_F)(const Fl_Label*, int,int,int,int, Fl_Align);

/** signature of some label measurement functions passed as parameters */
typedef void (Fl_Label_Measure_F)(const Fl_Label*, int&, int&);

/** signature of some box drawing functions passed as parameters */
typedef void (Fl_Box_Draw_F)(int,int,int,int, Fl_Color);

/** signature of some timeout callback functions passed as parameters */
typedef void (*Fl_Timeout_Handler)(void*);

/** signature of some wakeup callback functions passed as parameters */
typedef void (*Fl_Awake_Handler)(void*);

/**
  The Fl is the FLTK global (static) containing
  state information and global methods for the current application.
*/
class FL_EXPORT Fl {
  Fl() {}; // no constructor!

public: // should be private!
#ifndef FL_DOXYGEN
  static int e_number;
  static int e_x;
  static int e_y;
  static int e_x_root;
  static int e_y_root;
  static int e_dx;
  static int e_dy;
  static int e_state;
  static int e_clicks;
  static int e_is_click;
  static int e_keysym;
  static char* e_text;
  static int e_length;
  static Fl_Widget* belowmouse_;
  static Fl_Widget* pushed_;
  static Fl_Widget* focus_;
  static int damage_;
  static Fl_Widget* selection_owner_;
  static Fl_Window* modal_;
  static Fl_Window* grab_;
  static int compose_state;
  static int visible_focus_;
  static int dnd_text_ops_;
#endif
  /**
    If true then flush() will do something.
  */
  static void damage(int d) {damage_ = d;}

  /**
    The currently executing idle callback function: DO NOT USE THIS DIRECTLY!
    
    This is now used as part of a higher level system allowing multiple
    idle callback functions to be called.
    \see add_idle(), remove_idle()
  */
  static void (*idle)();

#ifndef FL_DOXYGEN
  static Fl_Awake_Handler *awake_ring_;
  static void **awake_data_;
  static int awake_ring_size_;
  static int awake_ring_head_;
  static int awake_ring_tail_;
  static const char* scheme_;
  static Fl_Image* scheme_bg_;

  static int e_original_keysym; // late addition
#endif


  static int add_awake_handler_(Fl_Awake_Handler, void*);
  static int get_awake_handler_(Fl_Awake_Handler&, void*&);

public:

  // API version number
  static double version();

  // argument parsers:
  static int arg(int, char**, int&);
  static int args(int, char**, int&, int (*)(int,char**,int&) = 0);
  static void args(int, char**);
  /**
    Usage string displayed if Fl::args() detects an invalid argument.
    This may be changed to point to customized text at run-time.
  */
  static const char* const help;

  // things called by initialization:
  static void display(const char*);
  static int visual(int);
  /**
    This does the same thing as
    Fl::visual(int) but also
    requires OpenGL drawing to work.  This <I>must</I> be done if
    you want to draw in normal windows with OpenGL with  gl_start() and
    gl_end().  It may be useful to call this so your X
    windows use the same visual as an
    Fl_Gl_Window, which on
    some servers will reduce colormap flashing.
    
    <P>See Fl_Gl_Window
    for a list of additional values for the argument.
  */
  static int gl_visual(int, int *alist=0); // platform dependent
  static void own_colormap();
  static void get_system_colors();
  static void foreground(uchar, uchar, uchar);
  static void background(uchar, uchar, uchar);
  static void background2(uchar, uchar, uchar);

  // schemes:
  static int scheme(const char*);
  /** See void scheme(const char *name) */
  static const char* scheme() {return scheme_;}
  /** 
    Called by scheme according to scheme name. 
    Loads or reloads the current scheme selection. 
    See void scheme(const char *name) 
  */
  static int reload_scheme(); // platform dependent
  static int scrollbar_size();
  static void scrollbar_size(int W);

  // execution:
  static int wait();
  static double wait(double time);
  static int check();
  static int ready();
  static int run();
  static Fl_Widget* readqueue();
  /**
  Adds a one-shot timeout callback.  The function will be called by
  Fl::wait() at <i>t</i> seconds after this function is called.
  The optional void* argument is passed to the callback.
  
  <P>You can have multiple timeout callbacks. To remove an timeout
  callback use Fl::remove_timeout().
  
  <p>If you need more accurate, repeated timeouts, use Fl::repeat_timeout() to
  reschedule the subsequent timeouts.</p>
  
  <p>The following code will print "TICK" each second on
  stdout with a fair degree of accuracy:</p>
  
  \code
     void callback(void*) {
       puts("TICK");
       Fl::repeat_timeout(1.0, callback);
     }
  
     int main() {
       Fl::add_timeout(1.0, callback);
       return Fl::run();
     }
  \endcode
  */
  static void add_timeout(double t, Fl_Timeout_Handler,void* = 0); // platform dependent
  /**
  This method repeats a timeout callback from the expiration of the
  previous timeout, allowing for more accurate timing. You may only call
  this method inside a timeout callback.
  
  <p>The following code will print "TICK" each second on
  stdout with a fair degree of accuracy:</p>
  
  \code
     void callback(void*) {
       puts("TICK");
       Fl::repeat_timeout(1.0, callback);
     }
  
     int main() {
       Fl::add_timeout(1.0, callback);
       return Fl::run();
     }
  \endcode
  */
  static void repeat_timeout(double t, Fl_Timeout_Handler,void* = 0); // platform dependent
  static int  has_timeout(Fl_Timeout_Handler, void* = 0);
  static void remove_timeout(Fl_Timeout_Handler, void* = 0);
  static void add_check(Fl_Timeout_Handler, void* = 0);
  static int  has_check(Fl_Timeout_Handler, void* = 0);
  static void remove_check(Fl_Timeout_Handler, void* = 0);
  /**
    Add file descriptor fd to listen to.  When the fd
    becomes ready for reading Fl::wait() will call the callback
    and then return.  The callback is
    passed the fd and the arbitrary void* argument.</P>
    
    <P>The second version takes a when bitfield, with the bits
    FL_READ, FL_WRITE, and FL_EXCEPT defined,
    to indicate when the callback should be done.
    
    <P>There can only be one callback of each type for a file descriptor. 
    Fl::remove_fd() gets rid of <I>all</I> the callbacks for a given
    file descriptor.
    
    <P>Under UNIX <I>any</I> file descriptor can be monitored (files,
    devices, pipes, sockets, etc.) Due to limitations in Microsoft Windows,
    WIN32 applications can only monitor sockets.
  */
  static void add_fd(int fd, int when, void (*cb)(int,void*),void* =0); // platform dependent
  /** See void add_fd(int fd, int when, void (*cb)(int,void*),void* =0) */
  static void add_fd(int fd, void (*cb)(int, void*), void* = 0); // platform dependent
  /** Removes a file descriptor handler. */
  static void remove_fd(int, int when); // platform dependent
  /** Removes a file descriptor handler. */
  static void remove_fd(int); // platform dependent

  static void add_idle(void (*cb)(void*), void* data = 0);
  static int  has_idle(void (*cb)(void*), void* data = 0);
  static void remove_idle(void (*cb)(void*), void* data = 0);
  /** If true then flush() will do something. */
  static int damage() {return damage_;}
  static void redraw();
  static void flush();
  /** \addtogroup group_comdlg
    @{ */
  /**
  FLTK calls this to print a warning message. You can
  override the behavior by setting the function pointer to your
  own routine.
  <P>Fl::warning means that there was a recoverable
  problem, the display may be messed up but the user can probably
  keep working - all X protocol errors call this, for example.
  */
  static void (*warning)(const char*, ...);
  /**
  FLTK calls this to print a normal error message. You can
  override the behavior by setting the function pointer to your
  own routine.
  <P>Fl::error means there is a recoverable error such as
  the inability to read an image file. The default implementation
  shows an error dialog to stderr and returns.
  */
  static void (*error)(const char*, ...);
  /**
  FLTK calls this to print a fatal error message. You can
  override the behavior by setting the function pointer to your
  own routine.
  <P>Fl::fatal must not return, as FLTK is in an unusable
  state, however your version may be able to use longjmp
  or an exception to continue, as long as it does not call FLTK
  again. The default implementation shows a fatal dialog box
  and exits with status 1.
  */
  static void (*fatal)(const char*, ...);
  /** @} */

  /** \defgroup  fl_windows Windows handling functions
      Windows and standard dialogs handling
    @{ */
  static Fl_Window* first_window();
  static void first_window(Fl_Window*);
  static Fl_Window* next_window(const Fl_Window*);

  /**
    Returns the top-most modal() window currently shown.
    This is the most recently 
    shown() window with 
    modal() true, or NULL if there are no modal()
    windows shown().
    The modal() window has its handle() method called
    for all events, and no other windows will have handle()
    called (grab() overrides this).
  */
  static Fl_Window* modal() {return modal_;}
  /**
    This is used when pop-up menu systems are active.  Send all events to
    the passed window no matter where the pointer or focus is (including
    in other programs).  The window <I>does not have to be
    shown()</I> , this lets the handle() method of a
    "dummy" window override all event handling and allows you to
    map and unmap a complex set of windows (under both X and WIN32
    <I>some</I> window must be mapped because the system interface needs a
    window id).
    
    <P>If grab() is on it will also affect show() of windows by
    doing system-specific operations (on X it turns on
    override-redirect).  These are designed to make menus popup reliably
    and faster on the system.
    
    <P>To turn off grabbing do Fl::grab(0).
    
    <P><I>Be careful that your program does not enter an infinite loop
    while grab() is on.  On X this will lock up your screen!</I>
    To avoid this potential lockup, all newer operating systems seem to 
    limit mouse pointer grabbing to the time during which a mouse button 
    is held down. Some OS's may not support grabbing at all.
  */
  static Fl_Window* grab() {return grab_;}
  /** Selects the window to grab. See Fl_Window* Fl::grab() */
  static void grab(Fl_Window*); // platform dependent
  /** @} */

  /** \defgroup fl_events Events handling functions
	Fl class events handling API
	@{
  */
  // event information:
  /**
    Returns the last event that was processed. This can be used
    to determine if a callback is being done in response to a
    keypress, mouse click, etc.
  */
  static int event()		{return e_number;}
  /**
    Returns the mouse position of the event relative to the Fl_Window
    it was passed to.
  */
  static int event_x()	{return e_x;}
  /**
    Returns the mouse position of the event relative to the Fl_Window
    it was passed to.
  */
  static int event_y()	{return e_y;}
  /**
    Returns the mouse position on the screen of the event.  To find the
    absolute position of an Fl_Window on the screen, use the
    difference between event_x_root(),event_y_root() and 
    event_x(),event_y().
  */
  static int event_x_root()	{return e_x_root;}
  /**
    Returns the mouse position on the screen of the event.  To find the
    absolute position of an Fl_Window on the screen, use the
    difference between event_x_root(),event_y_root() and 
    event_x(),event_y().
  */
  static int event_y_root()	{return e_y_root;}
  /**
    Returns the current horizontal mouse scrolling associated with the
    FL_MOUSEWHEEL event. Right is positive.
  */
  static int event_dx()	{return e_dx;}
  /**
    Returns the current vertical mouse scrolling assoaciated with the
    FL_MOUSEWHEEL event. Down is positive.
  */
  static int event_dy()	{return e_dy;}
  /**
    Return where the mouse is on the screen by doing a round-trip query to
    the server.  You should use Fl::event_x_root() and 
    Fl::event_y_root() if possible, but this is necessary if you are
    not sure if a mouse event has been processed recently (such as to
    position your first window).  If the display is not open, this will
    open it.
  */
  static void get_mouse(int &,int &); // platform dependent
  /**
    Returns non zero if we had a double click event.
    \retval Non-zero if the most recent FL_PUSH or FL_KEYBOARD was a "double click".  
    \retval  N-1 for  N clicks. 
    A double click is counted if the same button is pressed
    again while event_is_click() is true.
    
   */
  static int event_clicks()	{return e_clicks;}
  /**
    Manually sets the number returned by Fl::event_clicks().  
    This can be used to set it to zero so that
    later code does not think an item was double-clicked.
    \param[in] i correspond to no double-click if 0, i+1 mouse clicks otherwise
    \see int event_clicks()
  */
  static void event_clicks(int i) {e_clicks = i;}
  /**
    The first form returns non-zero if the mouse has not moved far enough
    and not enough time has passed since the last FL_PUSH or 
    FL_KEYBOARD event for it to be considered a "drag" rather than a
    "click".  You can test this on FL_DRAG, FL_RELEASE,
    and FL_MOVE events.  The second form clears the value returned
    by Fl::event_is_click().  Useful to prevent the <I>next</I>
    click from being counted as a double-click or to make a popup menu
    pick an item with a single click.  Don't pass non-zero to this.
  */
  static int event_is_click()	{return e_is_click;}
  /**
    Only i=0 works! See int event_is_click(). 
  */
  static void event_is_click(int i) {e_is_click = i;}
  /**
    Gets which particular mouse button caused the current event. 
    This returns garbage if the most recent event was not a FL_PUSH or FL_RELEASE event.
    \retval FL_LEFT_MOUSE \retval FL_MIDDLE_MOUSE \retval FL_RIGHT_MOUSE.
    \see Fl::event_buttons()
  */
  static int event_button()	{return e_keysym-FL_Button;}
  /**
    This is a bitfield of what shift states were on and what mouse buttons
    were held down during the most recent event.  The second version
    returns non-zero if any of the passed bits are turned on.  The legal
    bits are:
    
    <UL>
    <LI>FL_SHIFT</LI>
    <LI>FL_CAPS_LOCK</LI>
    <LI>FL_CTRL</LI>
    <LI>FL_ALT</LI>
    <LI>FL_NUM_LOCK</LI>
    <LI>FL_META</LI>
    <LI>FL_SCROLL_LOCK</LI>
    <LI>FL_BUTTON1</LI>
    <LI>FL_BUTTON2</LI>
    <LI>FL_BUTTON3</LI>
    </UL>
    
    <P>X servers do not agree on shift states, and FL_NUM_LOCK, FL_META, and
    FL_SCROLL_LOCK may not work.  The values were selected to match the
    XFree86 server on Linux.  In addition there is a bug in the way X works
    so that the shift state is not correctly reported until the first event <I>
    after</I> the shift key is pressed or released.
  */
  static int event_state()	{return e_state;}
  /** See int event_state() */
  static int event_state(int i) {return e_state&i;}
  /** Gets which key on the keyboard was last pushed.
      \retval 0 if the last event was not a key press or release.
    \see int event_key(int) */
  static int event_key()	{return e_keysym;}
  /**
    If NumLock is deactivated, FLTK translates events from the 
    numeric keypad into the corresponding arrow key events. 
    event_key() returns the translated key code, whereas
    event_original_key() returns the keycode before 
    NumLock translation.
  */
  static int event_original_key(){return e_original_keysym;}
  /** 
    Returns true if the given \a key was held
    down (or pressed) <I>during</I> the last event.  This is constant until
    the next event is read from the server.
    
    <P>Fl::get_key(int) returns true if the given key is held down <I>
    now</I>.  Under X this requires a round-trip to the server and is <I>
    much</I> slower than Fl::event_key(int).
    
    <P>Keys are identified by the <I>unshifted</I> values. FLTK defines a
    set of symbols that should work on most modern machines for every key
    on the keyboard:
    
    <UL>
    <LI>All keys on the main keyboard producing a printable ASCII
    character use the value of that ASCII character (as though shift,
    ctrl, and caps lock were not on).  The space bar is 32. </LI>
    <LI>All keys on the numeric keypad producing a printable ASCII
    character use the value of that ASCII character plus FL_KP.
    The highest possible value is FL_KP_Last so you can
    range-check to see if something is  on the keypad. </LI>
    <LI>All numbered function keys use the number on the function key plus 
    FL_F.  The highest possible number is FL_F_Last, so you
    can range-check a value. </LI>
    <LI>Buttons on the mouse are considered keys, and use the button
    number (where the left button is 1) plus FL_Button. </LI>
    <LI>All other keys on the keypad have a symbol: FL_Escape,
    FL_BackSpace, FL_Tab, FL_Enter, FL_Print, FL_Scroll_Lock, FL_Pause,
    FL_Insert, FL_Home, FL_Page_Up, FL_Delete, FL_End, FL_Page_Down,
    FL_Left, FL_Up, FL_Right, FL_Down, FL_Shift_L, FL_Shift_R,
    FL_Control_L, FL_Control_R, FL_Caps_Lock, FL_Alt_L, FL_Alt_R,
    FL_Meta_L, FL_Meta_R, FL_Menu, FL_Num_Lock, FL_KP_Enter.  Be
    careful not to confuse these with the very similar, but all-caps,
    symbols used by Fl::event_state()
    . </LI>
    </UL>
    
    <P>On X Fl::get_key(FL_Button+n) does not work.
    
    <P>On WIN32 Fl::get_key(FL_KP_Enter) and 
    Fl::event_key(FL_KP_Enter) do not work.
  */
  static int event_key(int key);
  /** 
    Returns true if the given \a key is held down <I>now</I>.  
    Under X this requires a round-trip to the server and is <I>
    much</I> slower than Fl::event_key(int). \see event_key(int)
  */
  static int get_key(int key); // platform dependent
  /** Returns the text associated with the current FL_PASTE or FL_DND_RELEASE event. */
  static const char* event_text() {return e_text;}
  /**
    Returns the length of the text in Fl::event_text().  There
    will always be a nul at this position in the text.  However there may
    be a nul before that if the keystroke translates to a nul character or
    you paste a nul character.
  */
  static int event_length() {return e_length;}

  static int compose(int &del);
  /**
    If the user moves the cursor, be sure to call Fl::compose_reset().
    The next call to Fl::compose() will start out in an initial state.  In
    particular it will not set "del" to non-zero. This call is very fast
    so it is ok to call it many times and in many places.
  */
  static void compose_reset() {compose_state = 0;}
  static int event_inside(int,int,int,int);
  static int event_inside(const Fl_Widget*);
  static int test_shortcut(int);

  // event destinations:
  static int handle(int, Fl_Window*);
  /**  Gets the widget that is below the mouse. 
       \see  belowmouse(Fl_Widget*) */
  static Fl_Widget* belowmouse() {return belowmouse_;}
  static void belowmouse(Fl_Widget*);
  /** Sets the widget that is being pushed 
      \see Fl_Widget* pushed(Fl_Widget*) */
  static Fl_Widget* pushed()	{return pushed_;}
  static void pushed(Fl_Widget*);
  /** Gets the current Fl::focus() widget. \sa Fl::focus(Fl_Widget*) */
  static Fl_Widget* focus()	{return focus_;}
  static void focus(Fl_Widget*);
  static void add_handler(int (*h)(int));
  static void remove_handler(int (*h)(int));
  /** @} */

  /** \defgroup  fl_clipboard  Selection & Clipboard functions
	fl global copy/cut/paste functions
   @{ */
  // cut/paste:
  /**
    Copies the data pointed to by stuff to the selection
    (0) or primary (1) clipboard.  The selection clipboard is used
    for middle-mouse pastes and for drag-and-drop selections. The
    primary clipboard is used for traditional copy/cut/paste
    operations.
  */
  static void copy(const char* stuff, int len, int clipboard = 0); // platform dependent
  /**
    Pastes the data from the selection (0) or primary (1) clipboard into receiver.  
    The selection clipboard is used for middle-mouse pastes and for 
    drag-and-drop selections. The primary clipboard is used for 
    traditional copy/cut/paste  operations.
  */
  static void paste(Fl_Widget &receiver, int clipboard /*=0*/); // platform dependent
  /**
    Initiate a Drag And Drop operation. The clipboard should be
    filled with relevant data before calling this method. FLTK will
    then initiate the system wide drag and drop handling. Dropped data
    will be marked as <i>text</i>.
  */
  static int dnd(); // platform dependent

  // These are for back-compatibility only:
  /**  back-compatibility only: Gets the widget owning the current selection  
       \see Fl_Widget* selection_owner(Fl_Widget*) */
  static Fl_Widget* selection_owner() {return selection_owner_;}
  static void selection_owner(Fl_Widget*);
  static void selection(Fl_Widget &owner, const char*, int len);
  static void paste(Fl_Widget &receiver);
/** @} */
/** \defgroup  fl_screen  Screen functions
	fl global screen functions
     @{ */
  // screen size:
  /** Returns the origin of the current screen, where 0 indicates the left side of the screen. */
  static int x(); // platform dependent
  /** Returns the origin of the current screen, where 0 indicates the top edge of the screen.  */
  static int y(); // platform dependent
  /** Returns the width of the screen in pixels. */
  static int w(); // platform dependent
  /** Returns the height of the screen in pixels. */
  static int h(); // platform dependent

  // multi-head support:
  static int screen_count();
  /** 
      Gets the bounding box of a screen 
      that contains the mouse pointer in.
      \param[out]  X,Y,W,H the corresponding screen bounding box
      \see void screen_xywh(int &x, int &y, int &w, int &h, int mx, int my) 
  */
  static void screen_xywh(int &X, int &Y, int &W, int &H) {
    screen_xywh(X, Y, W, H, e_x_root, e_y_root);
  }
  static void screen_xywh(int &X, int &Y, int &W, int &H, int mx, int my);
  static void screen_xywh(int &X, int &Y, int &W, int &H, int n);

  /**   @} */

  /** \defgroup  fl_attributes  Color & Font functions
	fl global color, font functions
     @{ */

  // color map:
  static void	set_color(Fl_Color, uchar, uchar, uchar);
  /**
    Sets an entry in the fl_color index table.  You can set it to
    any 8-bit RGB color.  The color is not allocated until fl_color(i)
    is used.
  */
  static void	set_color(Fl_Color, unsigned); // platorm dependent
  static unsigned get_color(Fl_Color);
  static void	get_color(Fl_Color, uchar&, uchar&, uchar&);
  /**
    Frees the specified color from the colormap, if applicable.
    If overlay is non-zero then the color is freed from the
    overlay colormap.
  */
  static void	free_color(Fl_Color, int overlay = 0); // platform dependent

  // fonts:
  static const char* get_font(Fl_Font);
  /**
    Get a human-readable string describing the family of this face.  This
    is useful if you are presenting a choice to the user.  There is no
    guarantee that each face has a different name.  The return value points
    to a static buffer that is overwritten each call.
    
    <P>The integer pointed to by attributes (if the pointer is not
    zero) is set to zero, FL_BOLD or FL_ITALIC or 
    FL_BOLD | FL_ITALIC.  To locate a "family" of fonts, search
    forward and back for a set with non-zero attributes, these faces along
    with the face with a zero attribute before them constitute a family.
  */
  static const char* get_font_name(Fl_Font, int* attributes = 0);
  /**
    Return an array of sizes in sizep.  The return value is the
    length of this array.  The sizes are sorted from smallest to largest
    and indicate what sizes can be given to fl_font() that will
    be matched exactly (fl_font() will pick the closest size for
    other sizes).  A zero in the first location of the array indicates a
    scalable font, where any size works, although the array may list sizes
    that work "better" than others.  Warning: the returned array
    points at a static buffer that is overwritten each call.  Under X this
    will open the display.
  */
  static int get_font_sizes(Fl_Font, int*& sizep);
  static void set_font(Fl_Font, const char*);
  static void set_font(Fl_Font, Fl_Font);
  /**
    FLTK will open the display, and add every fonts on the server to the
    face table.  It will attempt to put "families" of faces together, so
    that the normal one is first, followed by bold, italic, and bold
    italic.
    
    <P>The optional argument is a string to describe the set of fonts to
    add.  Passing NULL will select only fonts that have the
    ISO8859-1 character set (and are thus usable by normal text).  Passing
    "-*" will select all fonts with any encoding as long as they have
    normal X font names with dashes in them.  Passing "*" will list every
    font that exists (on X this may produce some strange output).  Other
    values may be useful but are system dependent.  With WIN32 NULL
    selects fonts with ISO8859-1 encoding and non-NULL selects
    all fonts.
    
    <P>The return value is how many faces are in the table after this is
    done.
  */
  static Fl_Font set_fonts(const char* = 0); // platform dependent

  /**   @} */

  // labeltypes:
  static void set_labeltype(Fl_Labeltype,Fl_Label_Draw_F*,Fl_Label_Measure_F*);
  /** Sets the functions to call to draw and measure a specific labeltype. */
  static void set_labeltype(Fl_Labeltype, Fl_Labeltype from); // is it defined ?

  // boxtypes:
  static Fl_Box_Draw_F *get_boxtype(Fl_Boxtype);
  static void set_boxtype(Fl_Boxtype, Fl_Box_Draw_F*,uchar,uchar,uchar,uchar);
  static void set_boxtype(Fl_Boxtype, Fl_Boxtype from);
  static int box_dx(Fl_Boxtype);
  static int box_dy(Fl_Boxtype);
  static int box_dw(Fl_Boxtype);
  static int box_dh(Fl_Boxtype);
  static int draw_box_active();

  // back compatibility:
  /** \addtogroup fl_windows 
    @{ */
  /** For back compatibility, sets the void Fl::fatal handler callback */
  static void set_abort(void (*f)(const char*,...)) {fatal = f;}
  static void (*atclose)(Fl_Window*,void*);
  static void default_atclose(Fl_Window*,void*);
  /** For back compatibility, sets the Fl::atclose handler callback. You
      can now simply change the callback for the window instead \see Fl_Window::callback(Fl_Callback*) */
  static void set_atclose(void (*f)(Fl_Window*,void*)) {atclose = f;}
  /**   @} */

  /** \addtogroup fl_events 
    @{ */
  /** Returns non-zero if the Shift key is pressed. */
  static int event_shift() {return e_state&FL_SHIFT;}
  /** Returns non-zero if the Control key is pressed. */
  static int event_ctrl() {return e_state&FL_CTRL;}
  /** Returns non-zero if the Alt key is pressed. */
  static int event_alt() {return e_state&FL_ALT;}
  /**
    Returns the mouse buttons state bits; if non-zero, then at least one
    button is pressed now.  This function returns the button state at the 
    time of the event. During an FL_RELEASE event, the state 
    of the released button will be 0. To find out, which button 
    caused an FL_RELEASE event, you can use Fl::event_button() instead.
    \return a bit mask value like { [FL_BUTTON1] | [FL_BUTTON2] | [FL_BUTTON3] }
  */
  static int event_buttons() {return e_state&0x7f000000;}
  /**
    Returns non-zero if mouse button 1 is currently held down.
    For more details, see Fl::event_buttons().
  */
  static int event_button1() {return e_state&FL_BUTTON1;}
  /**
    Returns non-zero if button 2 is currently held down.
    For more details, see Fl::event_buttons().
  */
  static int event_button2() {return e_state&FL_BUTTON2;}
  /**
    Returns non-zero if button 3 is currently held down.
    For more details, see Fl::event_buttons().
  */
  static int event_button3() {return e_state&FL_BUTTON3;}
  /**   @} */

  /**
    Sets an idle callback.
    <P>This method is obsolete - use the add_idle() method instead.
  */
  static void set_idle(void (*cb)()) {idle = cb;}
  /** See Fl_Window* grab() */
  static void grab(Fl_Window&win) {grab(&win);}
  /** Releases the current grabbed window, equals grab(0). See Fl_Window* grab() */
  static void release() {grab(0);}

  // Visible focus methods...
  /**
    Gets or sets the visible keyboard focus on buttons and other
    non-text widgets. The default mode is to enable keyboard focus
    for all widgets.
  */
  static void visible_focus(int v) { visible_focus_ = v; }
  /**
    Gets or sets the visible keyboard focus on buttons and other
    non-text widgets. The default mode is to enable keyboard focus
    for all widgets.
  */
  static int  visible_focus() { return visible_focus_; }

  // Drag-n-drop text operation methods...
  /**
    Gets or sets whether drag and drop text operations are
    supported. This specifically affects whether selected text can
    be dragged from text fields or dragged within a text field as a
    cut/paste shortcut.
  */
  static void dnd_text_ops(int v) { dnd_text_ops_ = v; }
  /**
    Gets or sets whether drag and drop text operations are
    supported. This specifically affects whether selected text can
    be dragged from text fields or dragged within a text field as a
    cut/paste shortcut.
  */
  static int  dnd_text_ops() { return dnd_text_ops_; }
  /** \defgroup fl_multithread Multithreading support functions
	fl multithreading support functions
   @{ */

  // Multithreading support:
  static void lock();
  static void unlock();
  static void awake(void* message = 0);
  /** See void awake(void* message=0). */
  static int awake(Fl_Awake_Handler cb, void* message = 0);
  /**
    The thread_message() method returns the last message
    that was sent from a child by the awake() method.
    
    <P>See also: multithreading
  */
  static void* thread_message(); // platform dependent
  /** @} */

  /** \defgroup fl_del_widget Safe widget deletion support functions
	fl multithreading support functions
   @{ */
  // Widget deletion:
  static void delete_widget(Fl_Widget *w);
  static void do_widget_deletion();
  static void watch_widget_pointer(Fl_Widget *&w);
  static void release_widget_pointer(Fl_Widget *&w);
  static void clear_widget_pointer(Fl_Widget const *w);
  /** @} */

#ifdef HAVE_CAIRO
  /** \defgroup group_cairo Cairo support functions and classes 
      @{ 
  */
public:
  // Cairo support API
  static cairo_t * cairo_make_current(Fl_Window* w);
   /** when HAVE_CAIRO is defined and cairo_autolink_context() is true, 
       any current window dc is linked to a current context.
      this is not the default because, it may not be necessary
      to add cairo support to all fltk supported windows.
      When you wish to associate a cairo context in this mode,
      you need to call explicitly in your draw() overriden method,
      FL::cairo_make_current(Fl_Window*). This will create a cairo context
      but only for this Window. 
      Still in custom cairo application it is possible to handle 
      completely this process automatically by setting \a alink to true.
      In this last case, you don't need anymore to call Fl::cairo_make_current()
      you can use Fl::cairo_cc() to get the current cairo context anytime.
     \note Only available when configure has the --enable-cairo option
  */
  static void cairo_autolink_context(bool alink) {cairo_state_.autolink(alink);}
  /** 
    Gets the current autolink mode for cairo support, 
    \retval false if no cairo context autolink is made for each window. 
    \retval true if any fltk window is attached a cairo context when it 
    is current. \see void cairo_autolink_context(bool alink) 
    \note Only available when configure has the --enable-cairo option
 */
  static bool cairo_autolink_context() {return cairo_state_.autolink();}
  /** Gets the current cairo context linked with a fltk window. */
  static cairo_t * cairo_cc() { return cairo_state_.cc(); }
  /** Sets the cairo current context to \a c, 
      set \a own to true if you want fltk to handle this cc deletion.
     \note Only available when configure has the --enable-cairo option
*/
  static void cairo_cc(cairo_t * c, bool own=false){ cairo_state_.cc(c, own); } 

private:
  static cairo_t * cairo_make_current(void* gc);
  static cairo_t * cairo_make_current(void* gc, int W, int H);
  static Fl_Cairo_State cairo_state_;
public:
  /** @} */ 

#endif // HAVE_CAIRO

};

#endif // !Fl_H

//
// End of "$Id: Fl.H 6439 2008-10-15 13:46:06Z fabien $".
//
