//
// fgfslog.cxx -- fgrun log cmd and flight, to
// (a) keep a text file of commands run, mainly for debug, and
// (b) a very rudimentary flight log
//
// Written by Geoff McLane, started June 2006.
// Copyright (c) 2006  Geoff McLane - geoffair at hotmail
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// module: fgfslog.cxx
// 2008-11-06 - minor update, using gmtime, in place of ctime  - grm

#include <string>
#include <iostream>
#include <stdio.h>
#include <time.h>

using namespace std;

// ========================================================
// functions to create and append to -
// (a) a command file, and
// (b) a flight log file
// version: 0.0.3 - 2008.12.06 - use gmtime, out YYYY-MM-DD HH:MM - geoff
// version: 0.0.2 - 2006.06.28 - added CSV output, and time difference - geoff
// version: 0.0.1 - 2006.06.24 - initial file - geoff
// ========================================================

// output file names - *TBD* make configurable
static char * logfile = "tempfgfs.log";
static char * cmdfile = "tempcmds.txt";

// *TBD* - these could also be configurable!
#define WRT_AS_CSV  // else as a flat database file
#define air_min_len     25  // used to space items, if NOT CSV
#define USE_GM_TIME // use gmtime to output YYYY-MM-DD HH:MM

#ifdef _MSC_VER
#define LOG_EOL     "\r\n"  // use WIN32 line endings
#define add_end_of_line(a)  fwrite( LOG_EOL, 1, 2, fh )
#else // !#ifdef _MSC_VER
#define LOG_EOL     "\n"
#define add_end_of_line(a)  fwrite( LOG_EOL, 1, 1, fh )
#endif // #ifdef _MSC_VER

#undef  DBG_TIMES   // just to provide some random debug values for DEBUG

#ifdef  WRT_AS_CSV
#define LOG_SEP_CHAR    ","
#else // !#ifdef  WRT_AS_CSV
#define LOG_SEP_CHAR    " "
#endif // #ifdef  WRT_AS_CSV y/n

static time_t log_curr_time, log_start_time;

static size_t _s_get_gm_time( char * buf, time_t t )
{
    struct tm * pt = gmtime(&t);
    size_t len;
    if(pt) {
        len = sprintf(buf, "%4d-%02d-%02d %02d:%02d",
            pt->tm_year + 1900,
            pt->tm_mon + 1,
            pt->tm_mday,
            pt->tm_hour,
            pt->tm_min);
    } else {
        len = sprintf(buf, "gmtime(%d) FAILED", t);
    }
    return len;

}

static void wrt_log_time( FILE * fh )
{
    log_curr_time = time(NULL);
#ifdef USE_GM_TIME
    char tim[32];
    size_t len = _s_get_gm_time( &tim[0], log_curr_time );
    fwrite( tim, 1, len, fh );
#else
    string s = ctime(&log_curr_time);
    size_t pos1 = s.find("\n"); // find LF,
    if( pos1 == string::npos ) { // if NOT FOUND
        pos1 = s.length();
    }
    fwrite( s.c_str(), 1, pos1, fh );
#endif
}

static string get_cmd_component( string & s, string & f, char * def )
{
    size_t pos1, pos2, sz;
    string are = " --"; // to next, if any
    string fnd = def;
    pos1 = s.find(f); // find,
    if( pos1 != string::npos ) { // if FOUND
        sz = pos1 + f.size(); // get the arg size
        pos2 = s.find( are, sz ); // find next arg beginning
        if( pos2 == string::npos ) { // if NOT FOUND
           pos2 = s.size();
        }
        pos1 += f.length(); // get to just the aircraft
        fnd = s.substr( pos1, pos2 - pos1 ); 
    }
    return fnd;
}

void out_fgfs_flight(string s)
{
    string air = "--aircraft=";
    string fdm = "--fdm=";
    FILE * fh;
    int add_hdr_line = 0;
    air = get_cmd_component( s, air, "c172" );
    // got an aircraft, or default ... find fdm command
    fdm = get_cmd_component( s, fdm, "jsb" );
    // got fdm, or default ... combine the two
    air += " (";
    air += fdm;
    air += ")"LOG_SEP_CHAR; // add separator
#ifndef  WRT_AS_CSV
    // up the length to the minimum
    while( air.length() < air_min_len )
        air += LOG_SEP_CHAR;
#endif // #ifdef  WRT_AS_CSV y/n
    // got aricraft and FDM strings
    fh = fopen( logfile, "r" ); // check if file exists ...
    if(fh) { // yup
        fclose(fh);
    } else { // no, add header line
        add_hdr_line = 1;
    }
    fh = fopen( logfile, "ab" ); // create or append (binary mode)
    if( fh ) {
        if(add_hdr_line) {
#ifdef  WRT_AS_CSV
            string hdr = "Aircraft (fdm)"LOG_SEP_CHAR
                "Start"LOG_SEP_CHAR
                "End"LOG_SEP_CHAR
                "Time"LOG_EOL;
#else // !#ifdef  WRT_AS_CSV
            string hdr = "Aircraft (fdm)";
            while(hdr.length() < air_min_len)
                hdr += " ";
            hdr += "Start";
            while(hdr.length() < air_min_len * 2)
                hdr += " ";
            hdr += "End";
            while(hdr.length() < air_min_len * 3)
                hdr += " ";
            hdr += "Time (HH:MM)"LOG_EOL;
#endif // #ifdef  WRT_AS_CSV y/n
            fwrite( hdr.c_str(), 1, hdr.length(), fh );
        }
        fwrite( air.c_str(), 1, air.length(), fh );
        wrt_log_time( fh );
        log_start_time = log_curr_time;

        fwrite( LOG_SEP_CHAR, 1, 1, fh );
        fclose(fh);
    } // fail gracefully
}

static void wrt_dif_time( FILE * fh, time_t diff )
{
    char tbuf[32];
    int mins = (int)(diff / 60);
    int hrs  = (int)(mins / 60);
    mins = mins - (hrs * 60);
    size_t len = sprintf(tbuf, LOG_SEP_CHAR"%02u:%02u", hrs, mins);
    fwrite( tbuf, 1, len, fh );
}

#ifdef  DBG_TIMES
time_t get_dbg_time( time_t diff )
{
    static int do_seed = 1;
    int RANGE_MIN = 0;
    int RANGE_MAX = 60;
    if( do_seed ) {
        unsigned int seed = (unsigned int)time(NULL);
        srand( seed );
        do_seed = 0;
    }
    int ih = (int)(((double) rand() / 
                 (double) RAND_MAX) * RANGE_MAX + RANGE_MIN);
    int im = (int)(((double) rand() / 
                 (double) RAND_MAX) * RANGE_MAX + RANGE_MIN);
    diff += ((ih * 60 * 60) + (im * 60));
    return diff;
}
#endif // #ifdef  DBG_TIMES


void add_fgfs_end( void )
{
    FILE * fh = fopen( logfile, "ab" ); // append
    if(fh) {
        wrt_log_time( fh ); // get and add current time
        time_t diff = log_curr_time - log_start_time; // difference
#ifdef  DBG_TIMES
        diff = get_dbg_time(diff); // just for DEBUG
#endif // #ifdef  DBG_TIMES
        wrt_dif_time( fh, diff ); // write difference as HH:MM
        add_end_of_line( fh );
        fclose(fh);
    } // fail gracefully
}

// just a DEBUG type list of the command lines
// passed to the system ...
void out_fgfs_run_cmd( char * cmd )
{
    size_t len = strlen(cmd);
    if(len) {
        FILE * fh = fopen( cmdfile, "ab" ); // append
        if( fh ) {
            fwrite( cmd, 1, len, fh );
            add_end_of_line( fh );
            fclose(fh);
        } // fail gracefully
    }
}
// ========================================================

// eof - fgfslog.cxx
