/*-------------------------------------------------------------------------
  win_utils.cxx

  Written by Geoff R. McLane, started January 2008.

  Copyright (C) 2008 Geoff R. McLane

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  ---------------------------------------------------------------------------*/

// win_utils.cxx
// Various minor utility functions for Atlas/Map suite of applications

#define ADD_WINDOW_DESTROY
#undef EXTRA_DEBUG_WIN_UTILS

#pragma warning(disable:4996)

#include <sys\types.h>
#include <sys\stat.h>   // for stat
#include "config.h"

#include <string>
#include <sstream>
#include <iostream>
#include <windows.h>    // for Sleep(ms)
#include <math.h>
#include <conio.h>      // for _kbhit() and _getch()
#include "win_utils.h"

using namespace std;

int check_key_available( void )
{
   int chr = _kbhit();
   return chr;
}

void make_beep_sound(void)
{
    //printf( 0x07 );
    //    freq.duration
    Beep( 750, 300 );
}

void win_terminate(void)
{
    // do nothing
}
// return TRUE if a valid path/file
// note: (windows) stat will FAIL on a path with a trailing '/',
// so the jiggery pokey stuff is to 'erase' a trailing '/', if it exists
// This is NOT a passed 'reference' so no harm is done to the passed path!
bool is_valid_path( std::string path )
{
    struct stat buf;
    size_t len = path.length();
    size_t pos = path.rfind("/");
    if ( pos == (len - 1) )
        path.erase( path.end() - 1 );
    if ( stat(path.c_str(),&buf) == 0 )
        return true;
    return false;
}

#if 0 // ==================================
extern int main_window, graphs_window;

void destroy_glut_windows(void)
{
#ifdef TRY_EXIT_GMAIN  // try using glutSetOption( GLUT_ACTION_ON_WINDOW_CLOSE
    glutSetOption( GLUT_ACTION_ON_WINDOW_CLOSE, GLUT_ACTION_GLUTMAINLOOP_RETURNS  );
#endif
#ifdef ADD_WINDOW_DESTROY
    if (graphs_window)
        glutDestroyWindow(graphs_window);
    if (main_window)
        glutDestroyWindow(main_window);
#endif
}

void clear_abort_behaviour(void)
{
    unsigned int i1, i2;
    //terminate_handler oldHand = set_terminate(win_terminate);
    // Suppress the abort message - but does not seem to work!!!
    i1 = _set_abort_behavior( 0, _WRITE_ABORT_MSG );
    i2 = _set_abort_behavior( 0, _CALL_REPORTFAULT );
#ifdef EXTRA_DEBUG_WIN_UTILS
    printf( "_set_abort_behavior returned %d and %d\n", i1, i2 );
#endif
    i1 = _set_abort_behavior( 0, _WRITE_ABORT_MSG );
    i2 = _set_abort_behavior( 0, _CALL_REPORTFAULT );
    if( i1 || i2 ) {
        printf( "_set_abort_behavior still returned %d and %d\n", i1, i2 );
    }
}

int wait_for_key(void)
{
    int chr;
    int sleep_for = 55; // ms
    long new_beep = 0;
    long next_beep = 15000;
    //std::cin >> c;
    while( !check_key_available() ) {
        new_beep += sleep_for;
        if (new_beep > next_beep) {
            make_beep_sound();
            new_beep = 0;
        } else
            Sleep(sleep_for);  // sleep for 55 ms
    }
    // a keyboard character is waiting - get it
    chr = _getch();
    if (chr < ' ' )
        chr += '@';
    return chr;
}

// special trap
void win_exit( int ret )
{
#if (defined(_MSC_VER) && defined(_DEBUG) && defined(ADD_MEMORY_DEBUG))
    show_crt_dbg_mem();
#endif // _MSC_VER and _DEBUG
#ifndef NDEBUG
    int chr;
    make_beep_sound();
    printf("Exit through win_exit(%d)...any KEY to exit : ", ret);
    chr = wait_for_key();
    printf( "%c - ", chr );
#else
    clear_abort_behaviour();
    destroy_glut_windows();
#endif
    if(ret) {
        exit(ret);
    }
#ifdef TRY_EXIT_GMAIN
    else
        glutLeaveMainLoop();
#else
    else
        exit(ret);
    //abort();
    //terminate();
#endif
#ifdef EXTRA_DEBUG_WIN_UTILS
    printf("Returning to glutMainLoop...\n");
#endif

}

#endif // 0 some excluded things

/* -----------------------
DESCRIPTION
The remainder () function computes the remainder of dividing x by y.
The return value is x " - " n " * " y, where n is the value x " / " y,
rounded to the nearest integer. If this quotient is 1/2 (mod 1), it is
rounded to the nearest even number (independent of the current rounding 
mode). If the return value is 0, it has the sign of x. The drem () function
does precisely the same thing. 
or
The remainder() function returns the remainder r: 
x - n * y;
where n is the integer nearest the exact value of x/y; if |n - x/y| = 1/2,
then n is even. 
   ----------------------- */

float remainderf( float x, float y )
{
   int n = (int)(( x / y ) + 0.5);
   float result = x - ( n * y );
   return result;
}

char * basename( char * name )
{
   size_t len = strlen(name);
   size_t i, j;
   int c;
   j = 0;
   for( i = 0; i < len; i++ ) {
      c = name[i];
      if(( c == '/' )||
         ( c == '\\')||
         ( c == ':' ))
      {
         j = i + 1;
      }
   }
   return &name[j];
}

void set_win_path_sep( std::string & s )
{
   size_t fnd;
   while( (fnd = s.find('/')) != -1 )
      s.replace(fnd, 1, "\\");
}

#if 0 // excluded stuff

double er_equ = 6378137.0;   // earth radius, equator (6,378.1370 km?)
double er_pol = 6356752.314; // earth radius, polar   (6,356.7523 km?)
double get_erad( double lat )
{
	// case SANSON_FLAMSTEED:
	double a = cos(lat) / er_equ;
	double b = sin(lat) / er_pol;
	return (1.0 / sqrt( a * a + b * b ));
}

void ll2pt( double obj_latr, double obj_lonr,
           double cent_latr, double cent_lonr,
           double map_size,  double map_zoom,
           double * px_pixel, double * py_pixel )
{
	double x,y,r;
	r = get_erad(cent_latr); // case SANSON_FLAMSTEED:
   x = r * cos(obj_latr)*(obj_lonr - cent_lonr);
   y = r * (obj_latr - cent_latr);
	*px_pixel = (map_size / 2.0) + (x * map_zoom);
	*py_pixel = (map_size / 2.0) + (y * map_zoom);
}

void pt2ll( double * pobj_latr, double * pobj_lonr,
           double cent_latr, double cent_lonr,
           double map_size,  double map_zoom,
           double x_pixel, double y_pixel )
{
	double x,y,r;
	r = get_erad(cent_latr); // case SANSON_FLAMSTEED:
	x = (x_pixel - (map_size / 2.0)) / map_zoom;
	y = (y_pixel - (map_size / 2.0)) / map_zoom;
      *pobj_latr = (y / r) + cent_latr;
      *pobj_lonr = (x / (r * cos(*pobj_latr))) + cent_lonr; 
}

bool  check_map_executable( std::string exe )
{
   bool failed = true;  // assume it will
   FILE *f;
   std::ostringstream cmd;
   std::string str = "";
   char c;
   size_t n;

   set_win_path_sep(exe);
   cmd << exe << " --version";
   if ((f = _popen(cmd.str().c_str(), "r")) == NULL) {
   } else {
      while (true) {
         n = fread(&c, 1, 1, f);
         if (n == 0) {
            if (feof(f)) {
               fclose(f);
               // check for 'version' is output by map
               if(( str.find(": version: ") != -1 )||   // found new version
                  ( str.find("FlightGear mapping utility") != -1 )) // or even OLD version
                  failed = false;
               break;
            }
         }
         str.append(1, c);
      }
   }
   if(failed) {
      if( str.length() > 1 )
         printf("WARNING: Got output '%s'\n", str.c_str() );
      printf("WARNING: The exe '%s' doesn't exist.\nAny missing maps can not be generated!\n",
         exe.c_str() );
      printf("Continue regardless? Enter y to continue, else will exit: ");
      std::cin >> c;
      if(( c == 'y' )||( c == 'Y' )) {
         printf( " continuing ...\n" );
         failed = false;
      } else {
         printf( " aborting ...\n" );
         ATLAS_EXIT(1);
      }
   }
   return failed;
}

std::string get_compiler_version_string( void )
{
   string s = ATLAS_COMPILER_STR;
   s += "\n";
   s += "Compiled on ";
   s += __DATE__;
   s += " at ";
   s += __TIME__;
   char * env = getenv("COMPUTERNAME");
   if(env) {
      s += "\n";
      s += "In ";
      s += env;
      env = getenv("USERNAME");
      if(env) {
         s += " by ";
         s += env;
      }
   }
   return s;
}

// stat - from : http://msdn.microsoft.com/en-us/library/14h5k7ff(VS.71).aspx
int jpg_or_png_exists( const char * pf, int base )
{
    static char _s_jp_buf[MAX_PATH];
    char * cp = _s_jp_buf;
    struct stat buf;
    if(base) {
        strcpy(cp,pf);
        buf.st_mode = 0;
        if (( stat(cp,&buf) == 0 )&& (buf.st_mode & _S_IFREG) )
            return 3;
    }
    strcpy(cp,pf);
    strcat(cp,".jpg");
    buf.st_mode = 0;
    if (( stat(cp,&buf) == 0 )&& (buf.st_mode & _S_IFREG) )
        return 1;
    strcpy(cp,pf);
    strcat(cp,".png");
    buf.st_mode = 0;
    if (( stat(cp,&buf) == 0 )&& (buf.st_mode & _S_IFREG) )
        return 2;
    return 0;
}

#ifdef _DEBUG
// ================================================
// debug memory - only works in _DEBUG mode
//int newFlag = _CRTDBG_ALLOC_MEM_DF | _CRTDBG_CHECK_CRT_DF | _CRTDBG_DELAY_FREE_MEM_DF |
//    _CRTDBG_LEAK_CHECK_DF;
static int newFlag = _CRTDBG_ALLOC_MEM_DF | _CRTDBG_DELAY_FREE_MEM_DF |  _CRTDBG_LEAK_CHECK_DF;
static int oldFlag = 0;

void set_crt_out_file( _HFILE hf )
{
   // Send all reports to STDOUT
   _CrtSetReportMode( _CRT_WARN, _CRTDBG_MODE_FILE );
   _CrtSetReportFile( _CRT_WARN, hf );
   _CrtSetReportMode( _CRT_ERROR, _CRTDBG_MODE_FILE );
   _CrtSetReportFile( _CRT_ERROR, hf );
   //_CrtSetReportFile( _CRT_ERROR, _CRTDBG_FILE_STDERR );
   _CrtSetReportMode( _CRT_ASSERT, _CRTDBG_MODE_FILE );
   _CrtSetReportFile( _CRT_ASSERT, hf );
}

void set_crt_dbg_mem(void)
{
    int tmpDbgFlag = _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG);
    tmpDbgFlag |= newFlag;
    oldFlag = _CrtSetDbgFlag( tmpDbgFlag );
    set_crt_out_file( _CRTDBG_FILE_STDOUT );
}

HANDLE get_win_file_handle( char * fname )
{
    HANDLE hand = CreateFile(
        fname,                          // LPCTSTR lpFileName,
        GENERIC_READ|GENERIC_WRITE,     // DWORD dwDesiredAccess,
        0,                              // DWORD dwShareMode,
        NULL,                           // LPSECURITY_ATTRIBUTES lpSecurityAttributes,
        CREATE_ALWAYS,                  // DWORD dwCreationDisposition,
        FILE_ATTRIBUTE_ARCHIVE,         // DWORD dwFlagsAndAttributes,
        NULL );                         // HANDLE hTemplateFile
    return hand;
}

void show_crt_dbg_mem(void)
{
    HANDLE hand;
    char * msg1 = "Dump of memory statistics...\n";
    char * msg2 = "Dump of memory objects...\n";
    _CrtMemState s1;
    _CrtMemCheckpoint( &s1 );
    hand = get_win_file_handle( "tempmemdbg.txt" );
    if ( hand && (hand != INVALID_HANDLE_VALUE) ) {
        DWORD dww;
        set_crt_out_file( hand );
        printf(msg1);
        WriteFile(hand, msg1, strlen(msg1), &dww, NULL);
        _CrtMemDumpStatistics( &s1 );
        printf(msg2);
        WriteFile(hand, msg2, strlen(msg2), &dww, NULL);
        _CrtMemDumpAllObjectsSince( NULL );
    } else {
        printf(msg1);
        _CrtMemDumpStatistics( &s1 );
        printf(msg2);
        _CrtMemDumpAllObjectsSince( NULL );
    }
}

#endif // _DEBUG
#endif // 0 exclude now

// eof - win_utils.cxx
